package cronapp.framework.authentication.social;

import com.google.gson.JsonObject;
import cronapp.framework.authentication.normal.AuthenticationConfigurer;
import cronapp.framework.authentication.token.AuthenticationController;
import cronapp.framework.authentication.token.AuthenticationResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.ResponseEntity;
import org.springframework.lang.Nullable;
import org.springframework.mobile.device.DeviceResolver;
import org.springframework.mobile.device.LiteDeviceResolver;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.social.connect.Connection;
import org.springframework.social.connect.UserProfile;
import org.springframework.social.connect.web.SignInAdapter;
import org.springframework.stereotype.Service;
import org.springframework.web.context.request.NativeWebRequest;

import javax.servlet.http.HttpServletRequest;
import java.util.Collections;

@Service
public class SocialSignInAdapter implements SignInAdapter {

  private static final Logger logger = LoggerFactory.getLogger(SocialSignInAdapter.class);

  private AuthenticationConfigurer authenticationConfigurer;

  private AuthenticationController authenticationController;

  private HttpServletRequest servletRequest;

  public SocialSignInAdapter(HttpServletRequest servletRequest,
                             @Nullable AuthenticationConfigurer authenticationConfigurer,
                             @Nullable AuthenticationController authenticationController) {
    this.authenticationConfigurer = authenticationConfigurer;
    this.authenticationController = authenticationController;
    this.servletRequest = servletRequest;
  }

  @Override
  public String signIn(String localUserId, Connection<?> connection, NativeWebRequest request) {
    logger.info("Social Sign Adaptor signIn() localUserId: " + localUserId);
    try {
      UserProfile userProfile = connection.fetchUserProfile();
      String email = userProfile.getEmail();

      if (email == null) {
        email = userProfile.getUsername();
      }
      logger.info("Social Sign Adaptor signIn() email: " + email);

      JsonObject json = new JsonObject();
      String userDisplayName = connection.getDisplayName();
      String userImageUrl = connection.getImageUrl();
      json.addProperty("name", userDisplayName);
      json.addProperty("image", userImageUrl);
      logger.info("Social Sign Adaptor signIn() name: " + userDisplayName);
      logger.info("Social Sign Adaptor signIn() image: " + userImageUrl);

      String provider = connection.getKey().getProviderId();

      UsernamePasswordAuthenticationToken auth = new UsernamePasswordAuthenticationToken(
          email, provider, Collections.singletonList(new SimpleGrantedAuthority("#OAUTH#")));

      auth.setDetails(json);

      try {
        if (authenticationConfigurer != null) {
          Authentication socialAuth = authenticationConfigurer.authenticate(auth);

          SecurityContextHolder.getContext().setAuthentication(socialAuth);

          servletRequest.getSession().setAttribute("#OAUTH#USER", email);
        } else {
          DeviceResolver deviceResolver = new LiteDeviceResolver();

          ResponseEntity<AuthenticationResponse> authenticationRequest = authenticationController.auth(email, provider, deviceResolver.resolveDevice(servletRequest), provider, null, json, servletRequest);

          AuthenticationResponse response = authenticationRequest.getBody();
          String token = response.getToken();

          logger.info("Social Sign Adaptor signIn() token: " + token);

          return servletRequest.getContextPath() + "/#/connected?_ctk=" + token;
        }
      } catch (Exception e) {
        logger.error("Social Sign Adaptor Error: Redirect to /#/notconnected due to: ", e);
        return servletRequest.getContextPath() + "/#/notconnected";
      }
    } catch (Exception e) {
      logger.error("Social Sign Adaptor Error: Redirect to /#/error/403 due to: ", e);
      return servletRequest.getContextPath() + "/#/error/403";
    }
    logger.error("Social Sign Adaptor Info: Redirect to /#/connected");
    return servletRequest.getContextPath() + "/#/connected";

  }
}
