package cronapp.framework.authentication.sso;

import com.google.gson.Gson;
import com.google.gson.JsonObject;
import cronapi.Var;
import cronapi.screen.Operations;
import cronapp.framework.api.ApiManager;
import cronapp.framework.api.EventsManager;
import cronapp.framework.authentication.security.CronappUserDetails;
import cronapp.framework.authentication.token.AuthenticationController;
import cronapp.framework.authentication.token.AuthenticationResponse;
import cronapp.framework.i18n.Messages;
import org.apache.commons.lang3.StringUtils;
import org.springframework.boot.autoconfigure.security.oauth2.client.OAuth2SsoProperties;
import org.springframework.boot.autoconfigure.security.oauth2.resource.PrincipalExtractor;
import org.springframework.boot.autoconfigure.security.oauth2.resource.UserInfoRestTemplateFactory;
import org.springframework.boot.autoconfigure.security.oauth2.resource.UserInfoTokenServices;
import org.springframework.context.ApplicationContext;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.mobile.device.DeviceResolver;
import org.springframework.mobile.device.LiteDeviceResolver;
import org.springframework.security.authentication.AuthenticationServiceException;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.config.annotation.SecurityConfigurerAdapter;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configurers.ExceptionHandlingConfigurer;
import org.springframework.security.oauth2.client.OAuth2RestOperations;
import org.springframework.security.oauth2.client.filter.OAuth2ClientAuthenticationProcessingFilter;
import org.springframework.security.oauth2.provider.OAuth2Authentication;
import org.springframework.security.oauth2.provider.authentication.OAuth2AuthenticationDetails;
import org.springframework.security.web.DefaultSecurityFilterChain;
import org.springframework.security.web.authentication.AuthenticationFailureHandler;
import org.springframework.security.web.authentication.AuthenticationSuccessHandler;
import org.springframework.security.web.authentication.HttpStatusEntryPoint;
import org.springframework.security.web.authentication.LoginUrlAuthenticationEntryPoint;
import org.springframework.security.web.authentication.preauth.AbstractPreAuthenticatedProcessingFilter;
import org.springframework.security.web.authentication.session.SessionAuthenticationStrategy;
import org.springframework.security.web.util.matcher.MediaTypeRequestMatcher;
import org.springframework.security.web.util.matcher.RequestHeaderRequestMatcher;
import org.springframework.web.accept.ContentNegotiationStrategy;
import org.springframework.web.accept.HeaderContentNegotiationStrategy;

import java.time.OffsetDateTime;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.UUID;

class SsoSecurityConfigurer {

  private final ApplicationContext applicationContext;

  SsoSecurityConfigurer(ApplicationContext applicationContext) {
    this.applicationContext = applicationContext;
  }

  void configure(HttpSecurity http) throws Exception {
    OAuth2SsoProperties sso = this.applicationContext.getBean(OAuth2SsoProperties.class);
    // Delay the processing of the filter until we know the SessionAuthenticationStrategy is available:
    http.apply(new OAuth2ClientAuthenticationConfigurer(oauth2SsoFilter(sso)));
    addAuthenticationEntryPoint(http, sso);
  }

  private void addAuthenticationEntryPoint(HttpSecurity http, OAuth2SsoProperties sso) throws Exception {
    ExceptionHandlingConfigurer<HttpSecurity> exceptions = http.exceptionHandling();
    ContentNegotiationStrategy contentNegotiationStrategy = http.getSharedObject(ContentNegotiationStrategy.class);
    if (contentNegotiationStrategy == null) {
      contentNegotiationStrategy = new HeaderContentNegotiationStrategy();
    }
    MediaTypeRequestMatcher preferredMatcher = new MediaTypeRequestMatcher(
        contentNegotiationStrategy, MediaType.APPLICATION_XHTML_XML,
        new MediaType("image", "*"), MediaType.TEXT_HTML, MediaType.TEXT_PLAIN);
    preferredMatcher.setIgnoredMediaTypes(Collections.singleton(MediaType.ALL));
    exceptions.defaultAuthenticationEntryPointFor(
        new LoginUrlAuthenticationEntryPoint(sso.getLoginPath()),
        preferredMatcher);
    // When multiple entry points are provided the default is the first one
    exceptions.defaultAuthenticationEntryPointFor(
        new HttpStatusEntryPoint(HttpStatus.UNAUTHORIZED),
        new RequestHeaderRequestMatcher("X-Requested-With", "XMLHttpRequest"));
  }

  private OAuth2ClientAuthenticationProcessingFilter oauth2SsoFilter(OAuth2SsoProperties sso) {
    OAuth2RestOperations restTemplate = this.applicationContext.getBean(UserInfoRestTemplateFactory.class).getUserInfoRestTemplate();
    UserInfoTokenServices tokenServices = this.applicationContext.getBean(UserInfoTokenServices.class);
    PrincipalExtractor principalExtractor = new CustomPrincipalExtractor();
    tokenServices.setAuthoritiesExtractor(new CustomAuthoritiesExtractor(principalExtractor));
    tokenServices.setPrincipalExtractor(principalExtractor);
    OAuth2ClientAuthenticationProcessingFilter filter = new OAuth2ClientAuthenticationProcessingFilter(sso.getLoginPath());
    filter.setRestTemplate(restTemplate);
    filter.setTokenServices(tokenServices);
    filter.setApplicationEventPublisher(this.applicationContext);
    filter.setAuthenticationSuccessHandler(successHandler());
    filter.setAuthenticationFailureHandler(failureHandler());
    return filter;
  }

  /**
   * Handler para sucesso de autorização
   */
  private AuthenticationSuccessHandler successHandler() {
    return (request, response, authentication) -> {
      UsernamePasswordAuthenticationToken userAuthentication = (UsernamePasswordAuthenticationToken) ((OAuth2Authentication) authentication).getUserAuthentication();

      CronappUserDetails externalUserDetails = getCronappUserDetails(userAuthentication);

      try {
        Gson gson = new Gson();
        JsonObject json = (JsonObject) gson.toJsonTree(externalUserDetails);

        DeviceResolver deviceResolver = new LiteDeviceResolver();

        AuthenticationController authenticationController = new AuthenticationController(null);

        String accessToken = String.valueOf(request.getAttribute(OAuth2AuthenticationDetails.ACCESS_TOKEN_VALUE));
        Operations.addTokenClaim(Var.valueOf("SSOAccessToken"), Var.valueOf(accessToken));

        ResponseEntity<AuthenticationResponse> authenticationRequest = authenticationController.auth(
            externalUserDetails.getEmail(),
            "cronapp",
            deviceResolver.resolveDevice(request),
            "SSO",
            null,
            json,
            request
        );

        String redirect = "/auth/signin/sso?_ctk=" + authenticationRequest.getBody().getToken();

        response.sendRedirect(redirect);
      } catch (Exception e) {
        throw new AuthenticationServiceException(Messages.getString("AuthError", e.getMessage()));
      }

      if (EventsManager.hasEvent("onLogin")) {
        EventsManager.executeEventOnTransaction("onLogin", Var.valueOf("username", authentication.getName()));
      }
    };
  }

  /**
   * Handler para falha de autorização
   */
  private AuthenticationFailureHandler failureHandler() {
    return (request, response, failureHandler) -> response.setStatus(HttpStatus.UNAUTHORIZED.value());
  }

  private String getAttributeString(Map<String, Object> details, String name) {
    if (details == null) {
      return null;
    }
    Object rawAttribute = details.get(name);
    if (rawAttribute == null) {
      return null;
    }
    return rawAttribute.toString();
  }

  @SuppressWarnings("unchecked")
  private CronappUserDetails getCronappUserDetails(UsernamePasswordAuthenticationToken userAuthentication) {
    Map<String, Object> details = (Map<String, Object>) userAuthentication.getDetails();

    String defaultPrincipal = String.valueOf(userAuthentication.getPrincipal());
    String userName = defaultPrincipal;
    String normalizedUserName = ApiManager.normalize(userName);
    String email = getAttributeString(details, "email");
    String normalizedEmail = ApiManager.normalize(email);
    String phone = getAttributeString(details, "phone");

    if (StringUtils.isEmpty(phone)) {
      phone = "N/A";
    }

    if (StringUtils.isEmpty(email)) {
      email = normalizedUserName + "@no-email";
    }

    if (StringUtils.isEmpty(normalizedEmail)) {
      normalizedEmail = ApiManager.normalize(email);
    }

    return CronappUserDetails.newBuilder()
        .setName(StringUtils.defaultIfEmpty(getAttributeString(details, "name"), userName))
        .setUserName(userName)
        .setNormalizedUserName(normalizedUserName)
        .setEmail(email)
        .setNormalizedEmail(normalizedEmail)
        .setEmailConfirmed(true)
        .setSecurityStamp(UUID.randomUUID().toString())
        .setPhoneNumber(phone)
        .setPhoneNumberConfirmed(true)
        .setTwoFactorEnabled(false)
        .setLockoutEnd(OffsetDateTime.MIN)
        .setLockoutEnabled(false)
        .setAccessFailedCount(0)
        .setAuthorities(Collections.unmodifiableSet(new LinkedHashSet<>(userAuthentication.getAuthorities())))
        .build();
  }

  private static class OAuth2ClientAuthenticationConfigurer
      extends SecurityConfigurerAdapter<DefaultSecurityFilterChain, HttpSecurity> {

    private OAuth2ClientAuthenticationProcessingFilter filter;

    OAuth2ClientAuthenticationConfigurer(OAuth2ClientAuthenticationProcessingFilter filter) {
      this.filter = filter;
    }

    @Override
    public void configure(HttpSecurity builder) {
      OAuth2ClientAuthenticationProcessingFilter ssoFilter = this.filter;
      ssoFilter.setSessionAuthenticationStrategy(builder.getSharedObject(SessionAuthenticationStrategy.class));
      builder.addFilterAfter(ssoFilter, AbstractPreAuthenticatedProcessingFilter.class);
    }
  }
}