package cronapp.framework.authentication.sso;

import cronapi.Var;
import cronapp.framework.api.ApiManager;
import cronapp.framework.api.EventsManager;
import cronapp.framework.authentication.AuthenticationUtil;
import cronapp.framework.authentication.security.CronappAnonymousAuthenticationFilter;
import cronapp.framework.authentication.security.FilterInvocationVoter;
import cronapp.framework.authentication.security.Permission;
import cronapp.framework.authentication.token.AuthenticationTokenFilter;
import org.apache.commons.lang3.StringUtils;
import org.json.JSONException;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.security.oauth2.OAuth2AutoConfiguration;
import org.springframework.boot.autoconfigure.security.oauth2.client.OAuth2SsoProperties;
import org.springframework.boot.autoconfigure.security.oauth2.resource.PrincipalExtractor;
import org.springframework.boot.autoconfigure.security.oauth2.resource.ResourceServerTokenServicesConfiguration;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.security.access.AccessDecisionVoter;
import org.springframework.security.access.vote.UnanimousBased;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter;
import org.springframework.security.config.http.SessionCreationPolicy;
import org.springframework.security.oauth2.client.OAuth2ClientContext;
import org.springframework.security.oauth2.client.OAuth2RestTemplate;
import org.springframework.security.oauth2.client.resource.OAuth2ProtectedResourceDetails;
import org.springframework.security.oauth2.config.annotation.web.configuration.EnableOAuth2Client;
import org.springframework.security.oauth2.config.annotation.web.configuration.OAuth2ClientConfiguration;
import org.springframework.security.web.authentication.Http403ForbiddenEntryPoint;
import org.springframework.security.web.authentication.UsernamePasswordAuthenticationFilter;
import org.springframework.security.web.authentication.logout.LogoutSuccessHandler;
import org.springframework.web.servlet.support.ServletUriComponentsBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.http.HttpServletResponse;
import java.net.URLEncoder;
import java.util.List;

@Configuration
@EnableOAuth2Client
@AutoConfigureBefore({OAuth2AutoConfiguration.class})
@EnableConfigurationProperties(OAuth2SsoProperties.class)
@Import({OAuth2ClientConfiguration.class, ResourceServerTokenServicesConfiguration.class})
public class AuthorizationConfigurer extends WebSecurityConfigurerAdapter {

  @Autowired(required = false)
  private Permission permission;

  @Autowired
  private ApplicationContext applicationContext;

  @Autowired
  private List<AccessDecisionVoter<? extends Object>> decisionVoters;

  @Value("${security.oauth2.client.customPrincipalKey:#{null}}")
  private String customPrincipalKey;

  @Value("${security.oauth2.client.logoutUri:#{null}}")
  private String logoutUri;

  private static final Logger logger = LoggerFactory.getLogger(AuthorizationConfigurer.class);

  @Bean
  public AuthenticationTokenFilter authenticationTokenFilterBean() throws Exception {
    AuthenticationTokenFilter authenticationTokenFilter = new AuthenticationTokenFilter();
    authenticationTokenFilter.setAuthenticationManager(super.authenticationManagerBean());
    return authenticationTokenFilter;
  }

  @Override
  protected void configure(HttpSecurity http) throws Exception {
    // post sem csrf
    http.csrf().disable();

    // security permission
    if (permission == null) {
      http.anonymous().authenticationFilter(new CronappAnonymousAuthenticationFilter("anonymousAuthenticationFilterKey", "anonymousUser", ApiManager.getPublicAuthorities()));
      http.authorizeRequests().anyRequest().denyAll().accessDecisionManager(new UnanimousBased(decisionVoters));
    } else {
      permission.loadSecurityPermission(http);
      AuthenticationUtil.loadStaticSecurity(http);
    }

    // logout
    http.exceptionHandling()
        .authenticationEntryPoint(new Http403ForbiddenEntryPoint())
        .and()
        .logout()
        .logoutSuccessHandler(logoutHandler())
        .deleteCookies("_u")
        .and().sessionManagement().sessionCreationPolicy(SessionCreationPolicy.STATELESS);

    // x-frame-options disable
    http.headers().cacheControl().disable().frameOptions().disable().httpStrictTransportSecurity().disable();

    new SsoSecurityConfigurer(applicationContext).configure(http);
    // Custom JWT based authentication
    http.addFilterBefore(authenticationTokenFilterBean(), UsernamePasswordAuthenticationFilter.class);
  }

  @Bean
  public OAuth2RestTemplate oauth2RestTemplate(OAuth2ClientContext oauth2ClientContext, OAuth2ProtectedResourceDetails details) {
    return new OAuth2RestTemplate(details, oauth2ClientContext);
  }

  @Bean
  public PrincipalExtractor customPrincipalExtractor() {
    System.setProperty("security.oauth2.client.customPrincipalKey", StringUtils.defaultIfBlank(customPrincipalKey, ""));
    return new CustomPrincipalExtractor();
  }

  @Bean
  public AccessDecisionVoter filterInvocationVoter() {
    return new FilterInvocationVoter();
  }

  private LogoutSuccessHandler logoutHandler() {
    return (request, response, authentication) -> {
      if (EventsManager.hasEvent("onLogout") && authentication != null && authentication.getName() != null) {
        EventsManager.executeEventOnTransaction("onLogout", Var.valueOf("username", authentication.getName()));
      }

      String appBaseUrl = ServletUriComponentsBuilder.fromRequestUri(request).replacePath(null).build().toUriString();

      if (StringUtils.isNotBlank(logoutUri)) {
        logoutUri = logoutUri.replace("${appURL}", appBaseUrl)
            .replace("${appURLEncoded}", URLEncoder.encode(appBaseUrl, "UTF-8"));
      } else {
        logoutUri = request.getContextPath() + "/index.html";
      }

      if (request.getHeader("Accept") == null || !request.getHeader("Accept").contains("json")) {
        response.setStatus(HttpServletResponse.SC_OK);
        response.sendRedirect(request.getContextPath() + "/index.html");
      } else {
        response.setContentType("application/json");

        String jsonString = "{}";
        try {
          jsonString = new JSONObject()
              .put("logoutUri", logoutUri)
              .put("appBaseUrl", appBaseUrl).toString();
        } catch (JSONException e) {
          logger.error("Could not create JSON Object in SSO logoutHandler method.", e);
        }

        response.getWriter().print(jsonString);
      }
    };
  }
}