package cronapp.framework.api;

import com.google.gson.JsonObject;
import cronapi.Var;
import cronapi.database.DataSource;
import cronapi.database.DatabaseQueryManager;
import cronapi.util.LRUCache;
import cronapp.framework.authentication.security.CronappUserDetails;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.time.DateUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;
import org.springframework.security.crypto.password.PasswordEncoder;

import java.text.Normalizer;
import java.time.Instant;
import java.util.*;
import java.util.stream.Collectors;

public final class ApiManager {

  public static final String SECURABLE_ATTRIBUTE_NAME = "name";
  public static final String SECURABLE_ATTRIBUTE_PATTERN = "pattern";
  public static final String SECURABLE_ATTRIBUTE_TYPE = "type";
  public static final String SECURABLE_TYPE_VIEW = "view";

  private static final Logger logger = LoggerFactory.getLogger(ApiManager.class);

  private static boolean OLD_AUTHORIZATION;

  private final String username;

  private final String password;

  private final String type;

  private final boolean autoSignUp;
  private static int INTERVAL = 1000 * 60;
  private final static LRUCache<String, Collection<Var>> SECURABLE_CACHE = new LRUCache<>(100, INTERVAL);

  private final PasswordEncoder passwordEncoder = new BCryptPasswordEncoder();
  private Boolean passwordMatches = false;
  private JsonObject details;

  static {
    try {
      Class.forName("auth.permission.SecurityPermission");
      OLD_AUTHORIZATION = true;
    } catch (Throwable e) {
      OLD_AUTHORIZATION = false;
    }
  }

  private ApiManager(String username, String password, String type, boolean autoSignUp, JsonObject details) {
    this.username = normalize(username);
    this.password = password;
    this.type = (type == null ? "local" : type);
    this.autoSignUp = autoSignUp;
    this.details = details;
    try {
      DatabaseQueryManager authManager = new DatabaseQueryManager("auth");
      passwordMatches = !EventsManager.hasEvent("onAuthenticate") && authManager.isDatabase();
    } catch (Exception e) {
      logger.error("It was not possible to authenticate.", e);
    }
  }

  public static ApiManager byUser(String username) {
    return new ApiManager(username, null, "local", false, null);
  }

  public static ApiManager byUserAndPassword(String username, String password, String type, boolean autoSignUp, JsonObject details) {
    return new ApiManager(username, password, type, autoSignUp, details);
  }

  public static ApiManager byUserAndPassword(String username, String password) {
    return new ApiManager(username, password, "local", false, null);
  }

  public static Collection<Var> getUserSecurables(String username) {

    Collection<Var> listSecurables;
    listSecurables = SECURABLE_CACHE.get("user-" + username);
    if (listSecurables != null) {
      return listSecurables;
    }
    listSecurables = new LinkedHashSet<>();

    username = normalize(username);
    Var securableResult;
    try {
      if (EventsManager.hasEvent("onGetUserSecurables")) {
        securableResult = EventsManager.executeEventOnTransaction("onGetUserSecurables", Var.valueOf(username));
        if (securableResult.getType() != Var.Type.NULL) {
          if (securableResult.getType() != Var.Type.LIST && !(securableResult.getObject() instanceof DataSource)) {
            listSecurables.add(securableResult);
          } else {
            for (Object securable : securableResult.getObjectAsList()) {
              listSecurables.add(Var.valueOf(securable));
            }
          }
        }
      } else {
        DatabaseQueryManager securableManager = new DatabaseQueryManager("userSecurables");
        securableResult = securableManager.get(username);

        if (securableManager.isDatabase()) {
          if (securableResult.size() > 0) {
            Var securableObject = securableResult.get(0);
            if (securableObject.isNative()) {
              for (Object userSecurable : securableResult.getObjectAsList()) {
                listSecurables.add(Var.valueOf(userSecurable));
              }
            } else {
              List userSecurables = securableResult.getObjectAsList();
              for (Object userSecurable : userSecurables) {
                listSecurables.add(Var.valueOf(userSecurable));
              }
            }
          }
        } else {
          if (securableResult.getType() != Var.Type.NULL) {
            if (securableResult.getType() != Var.Type.LIST) {
              listSecurables.add(securableResult);
            } else {
              for (Object securable : securableResult.getObjectAsList()) {
                listSecurables.add(Var.valueOf(securable));
              }
            }
          }
        }
      }
      if (!listSecurables.isEmpty()) {
        SECURABLE_CACHE.put("user-" + username, listSecurables);
      }
    } catch (Exception e) {
      logger.error(e.getMessage());
    }
    return listSecurables;
  }

  public static Collection<Var> getAuthenticatedSecurables() {
    return getRoleSecurables("authenticated users");
  }

  public static Collection<Var> getSecurables(String name) {
    LinkedHashSet<Var> listSecurables = new LinkedHashSet<>();
    Var securableResult;
    try {
      if (EventsManager.hasEvent("onGetSecurables")) {
        securableResult = EventsManager.executeEventOnTransaction("onGetSecurables", Var.valueOf(name));
        if (securableResult.getType() != Var.Type.NULL) {
          if (securableResult.getType() != Var.Type.LIST && !(securableResult.getObject() instanceof DataSource)) {
            listSecurables.add(securableResult);
          } else {
            for (Object securable : securableResult.getObjectAsList()) {
              listSecurables.add(Var.valueOf(securable));
            }
          }
        }
      } else {
        DatabaseQueryManager securableManager = new DatabaseQueryManager("securables");
        securableResult = securableManager.get(name);

        if (securableManager.isDatabase()) {
          if (securableResult.size() > 0) {
            Var securableObject = securableResult.get(0);
            if (securableObject.isNative()) {
              for (Object userSecurable : securableResult.getObjectAsList()) {
                listSecurables.add(Var.valueOf(userSecurable));
              }
            } else {
              List userSecurables = securableResult.getObjectAsList();
              for (Object userSecurable : userSecurables) {
                listSecurables.add(Var.valueOf(userSecurable));
              }
            }
          }
        } else {
          if (securableResult.getType() != Var.Type.NULL) {
            if (securableResult.getType() != Var.Type.LIST) {
              listSecurables.add(securableResult);
            } else {
              for (Object securable : securableResult.getObjectAsList()) {
                listSecurables.add(Var.valueOf(securable));
              }
            }
          }
        }
      }
    } catch (Exception e) {
      logger.error(e.getMessage());
    }
    return listSecurables;
  }

  public static Var getUserFromProvider(String providerName, String providerKey) {
    DatabaseQueryManager loginFromProvider = new DatabaseQueryManager("userProviderKey");
    Var providerResult;
    Var user = null;
    try {
      providerResult = loginFromProvider.get(providerName, providerKey);
      Var providerObject;
      if (providerResult != null) {
        if (providerResult.size() > 0) {
          providerObject = providerResult.get(0);
          user = providerObject.getField("user");
        }
      }
    } catch (Exception e) {
      logger.error(e.getMessage());
    }
    return user;
  }

  public static Collection<Var> getSecurableViews(String name) {
    Collection<Var> listSecurables;
    listSecurables = SECURABLE_CACHE.get("view-" + name);
    if (listSecurables != null) {
      return listSecurables;
    }
    listSecurables = new LinkedHashSet<>();
    Var securableResult;
    try {
      if (EventsManager.hasEvent("onGetSecurableViews")) {
        securableResult = EventsManager.executeEventOnTransaction("onGetSecurableViews", Var.valueOf(name));
        if (securableResult.getType() != Var.Type.NULL) {
          if (securableResult.getType() != Var.Type.LIST && !(securableResult.getObject() instanceof DataSource)) {
            listSecurables.add(securableResult);
          } else {
            for (Object securable : securableResult.getObjectAsList()) {
              listSecurables.add(Var.valueOf(securable));
            }
          }
        }
      } else {
        DatabaseQueryManager securableManager = new DatabaseQueryManager("securableViews");
        securableResult = securableManager.get(name);

        if (securableManager.isDatabase()) {
          if (securableResult.size() > 0) {
            Var securableObject = securableResult.get(0);
            if (securableObject.isNative()) {
              for (Object userSecurable : securableResult.getObjectAsList()) {
                listSecurables.add(Var.valueOf(userSecurable));
              }
            } else {
              List userSecurables = securableResult.getObjectAsList();
              for (Object userSecurable : userSecurables) {
                listSecurables.add(Var.valueOf(userSecurable));
              }
            }
          }
        } else {
          if (securableResult.getType() != Var.Type.NULL) {
            if (securableResult.getType() != Var.Type.LIST) {
              listSecurables.add(securableResult);
            } else {
              for (Object securable : securableResult.getObjectAsList()) {
                listSecurables.add(Var.valueOf(securable));
              }
            }
          }
        }
      }
      if (!listSecurables.isEmpty()) {
        SECURABLE_CACHE.put("view-" + name, listSecurables);
      }
    } catch (Exception e) {
      logger.error(e.getMessage());
    }
    return listSecurables;
  }

  public static List<GrantedAuthority> getPublicAuthorities() {
    return getPublicSecurables().stream()
        .map(securable -> securable.getStringField(ApiManager.SECURABLE_ATTRIBUTE_NAME))
        .map(SimpleGrantedAuthority::new)
        .collect(Collectors.toList());
  }

  public static Collection<Var> getPublicSecurables() {
    return getRoleSecurables("anonymous users");
  }

  public static Collection<Var> getRoleSecurables(String rolename) {
    rolename = normalize(rolename);
    Collection<Var> listSecurables;
    listSecurables = SECURABLE_CACHE.get("role-" + rolename);
    if (listSecurables != null) {
      return listSecurables;
    }
    listSecurables = new LinkedHashSet<>();
    Var securableResult;
    try {
      if (EventsManager.hasEvent("onGetRoleSecurables")) {
        securableResult = EventsManager.executeEventOnTransaction("onGetRoleSecurables", Var.valueOf(rolename));
        if (securableResult.getType() != Var.Type.NULL) {
          if (securableResult.getType() != Var.Type.LIST && !(securableResult.getObject() instanceof DataSource)) {
            listSecurables.add(securableResult);
          } else {
            for (Object securable : securableResult.getObjectAsList()) {
              listSecurables.add(Var.valueOf(securable));
            }
          }
        }
      } else {
        DatabaseQueryManager securableManager = new DatabaseQueryManager("roleSecurables");
        securableResult = securableManager.get(rolename);

        if (securableManager.isDatabase()) {
          if (securableResult.size() > 0) {
            Var securableObject = securableResult.get(0);
            if (securableObject.isNative()) {
              for (Object userSecurable : securableResult.getObjectAsList()) {
                listSecurables.add(Var.valueOf(userSecurable));
              }
            } else {
              List userSecurables = securableResult.getObjectAsList();
              for (Object userSecurable : userSecurables) {
                listSecurables.add(Var.valueOf(userSecurable));
              }
            }
          }
        } else {
          if (securableResult.getType() != Var.Type.NULL) {
            if (securableResult.getType() != Var.Type.LIST) {
              listSecurables.add(securableResult);
            } else {
              for (Object securable : securableResult.getObjectAsList()) {
                listSecurables.add(Var.valueOf(securable));
              }
            }
          }
        }
      }
      if (!listSecurables.isEmpty()) {
        SECURABLE_CACHE.put("role-" + rolename, listSecurables);
      }
    } catch (Exception e) {
      logger.error(e.getMessage());
    }
    return listSecurables;
  }

  public static String normalize(String s) {
    if (StringUtils.isEmpty(s)) {
      return "";
    }

    if (OLD_AUTHORIZATION) {
      return s;
    }

    return Normalizer
        .normalize(s, Normalizer.Form.NFD)
        .replaceAll("[^\\p{ASCII}]", "")
        .toLowerCase().trim();
  }

  public static void updateDevice(Device device) {
    try {
      logger.info("Device logged in:  " + device.getId());
      Var deviceResult;
      DatabaseQueryManager deviceManager = new DatabaseQueryManager("device");
      deviceResult = deviceManager.get(device.getId());
      if (deviceManager.isDatabase()) {
        if (deviceResult.size() > 0) {
          Var deviceObject = deviceResult.get(0);
          deviceObject.setField("token", device.getToken());
          deviceObject.setField("appVersion", device.getAppVersion());
          deviceObject.setField("platformVersion", device.getPlatformVersion());
          deviceManager.update(deviceObject);
        } else {
          Map<String, String> map = new LinkedHashMap<>();
          map.put("id", device.getId());
          map.put("platform", device.getPlatform());
          map.put("platformVersion", device.getPlatformVersion());
          map.put("appName", device.getAppName());
          map.put("appVersion", device.getAppVersion());
          map.put("token", device.getToken());
          map.put("model", device.getModel());
          deviceManager.insert(Var.valueOf(map));
        }
      }
    } catch (Exception e) {
      logger.error(e.getMessage(), e);
    }
  }

  public boolean passwordMatches(CharSequence rawPassword, String password) {
    if (passwordMatches) {
      return passwordEncoder.matches(rawPassword, password) || rawPassword.equals(password);
    } else {
      return true;
    }
  }

  public User getUser() throws Exception {
    return getUser(null);
  }

  private Var getField(Var obj, String... fields) {
    for (String field : fields) {
      Var value = obj.getField(field);
      if (value != null && value.getObject() != null && !value.getObjectAsString().isEmpty()) {
        return value;
      }
    }

    return Var.VAR_NULL;
  }

  public User getUser(CronappUserDetails userDetails) throws Exception {
    Var resultObject = this.getUserVar(userDetails);
    if (resultObject != null) {
      Var id = resultObject.getField("id");
      Var name = getField(resultObject, "name", "login", "username", "userName");
      Var username = getField(resultObject, "login", "username", "userName");
      Var password = resultObject.getField("password");
      Var theme = resultObject.getField("theme");
      Var picture = resultObject.getField("picture");
      Var lockoutEnabled = resultObject.getField("lockoutEnabled");
      Var lockoutEnd = resultObject.getField("lockoutEnd");
      Var accessFailedCount = resultObject.getField("accessFailedCount");
      return new User(id, name, username, password, theme, picture, lockoutEnabled, lockoutEnd, accessFailedCount);
    }
    return null;
  }

  private Var getUserVar(CronappUserDetails userDetails) throws Exception {
    Var userResult = null;
    if (EventsManager.hasEvent("onAuthenticate")) {
      userResult = EventsManager.executeEventOnTransaction("onAuthenticate", Var.valueOf(username), Var.valueOf(password), Var.valueOf(type), Var.valueOf(userDetails));
    } else {
      DatabaseQueryManager authManager = new DatabaseQueryManager("auth");
      userResult = authManager.get(username);

      if (userResult.getType() == Var.Type.LIST && userResult.size() == 0 && autoSignUp && !("local".equals(type))) {
        Map<String, Object> map = new LinkedHashMap<>();
        if (userDetails == null) {
          map.put("name", getDetail("name", username));
          map.put("userName", getDetail("login", username));
          map.put("normalizedUserName", ApiManager.normalize(getDetail("login", username)));
          map.put("normalizedEmail", ApiManager.normalize(getDetail("email", username)));
          map.put("login", getDetail("login", username));
          map.put("email", getDetail("email", username));

          map.put("image", getDetail("image", null));
          map.put("picture", getDetail("image", null));

          map.put("password", UUID.randomUUID().toString());
          map.put("theme", getDetail("theme", ""));
        } else {
          map.put("name", userDetails.getName());
          map.put("login", userDetails.getNormalizedUserName());
          map.put("email", userDetails.getEmail());

          map.put("password", UUID.randomUUID().toString());

          map.put("id", UUID.randomUUID());
          map.put("userName", userDetails.getUserName());
          map.put("normalizedUserName", userDetails.getNormalizedUserName());
          map.put("normalizedEmail", userDetails.getNormalizedEmail());
          map.put("emailConfirmed", userDetails.isEmailConfirmed());
          map.put("securityStamp", userDetails.getSecurityStamp());
          map.put("phoneNumber", userDetails.getPhoneNumber());
          map.put("phoneNumberConfirmed", userDetails.isPhoneNumberConfirmed());
          map.put("twoFactorEnabled", userDetails.isTwoFactorEnabled());
          map.put("lockoutEnd", userDetails.getLockoutEnd());
          map.put("lockoutEnabled", userDetails.isLockoutEnabled());
          map.put("accessFailedCount", userDetails.getAccessFailedCount());
          map.put("picture", getDetail("image", null));
        }
        userResult = Var.valueOf(map);
        authManager.insert(userResult);
      }

    }
    if (userResult.getType() == Var.Type.BOOLEAN) {
      if (userResult.getObjectAsBoolean()) {
        Map<String, String> map = new LinkedHashMap<>();
        map.put("login", username);
        map.put("password", password);
        map.put("theme", "");

        return Var.valueOf(map);
      }
    } else if (userResult.getType() == Var.Type.LIST && userResult.size() > 0) {
      return userResult.get(0);
    } else if (userResult.getType() == Var.Type.LIST && userResult.size() == 0) {
      return null;
    } else {
      if (userResult.getType() != Var.Type.NULL)
        return userResult;
    }
    return null;
  }

  private String getDetail(String key, String defValue) {
    if (details != null && details.has(key) && details.get(key) != null && !details.get(key).isJsonNull() && StringUtils.isBlank(details.get(key).getAsString())) {
      return details.get(key).getAsString();
    }

    return defValue;
  }

  public Collection<String> getRoles() {
    LinkedHashSet<String> listRoles = new LinkedHashSet<>();
    Var roleResult;
    try {
      if (EventsManager.hasEvent("onGetRoles")) {
        roleResult = EventsManager.executeEventOnTransaction("onGetRoles", Var.valueOf(username));
        if (roleResult.getType() != Var.Type.NULL) {
          if (roleResult.getType() != Var.Type.LIST && !(roleResult.getObject() instanceof DataSource)) {
            listRoles.add(roleResult.getObjectAsString());
          } else {
            for (Object role : roleResult.getObjectAsList()) {
              listRoles.add(Var.valueOf(role).getObjectAsString());
            }
          }
        }
      } else {
        DatabaseQueryManager roleManager = new DatabaseQueryManager("roles");
        roleResult = roleManager.get(username);

        if (roleManager.isDatabase()) {
          if (roleResult.size() > 0) {
            Var roleObject = roleResult.get(0);
            if (roleObject.isNative()) {
              for (Object userRole : roleResult.getObjectAsList()) {
                String roleString = Var.valueOf(userRole).getObjectAsString();
                listRoles.add(roleString);
              }
            } else {
              List userRoles = roleResult.getObjectAsList();
              for (Object userRole : userRoles) {
                Var roleVar = Var.valueOf(userRole).getField("role");
                Var roleName = roleVar.getField("name");
                String roleString = roleName.getObjectAsString();
                listRoles.add(roleString);
              }
            }
          }
        } else {
          if (roleResult.getType() != Var.Type.NULL) {
            if (roleResult.getType() != Var.Type.LIST) {
              listRoles.add(roleResult.getObjectAsString());
            } else {
              for (Object role : roleResult.getObjectAsList()) {
                listRoles.add(Var.valueOf(role).getObjectAsString());
              }
            }
          }
        }
      }
    } catch (Exception e) {
      logger.error(e.getMessage());
    }

    Collection<String> allSecurables = listRoles.stream()
        .flatMap(role -> getRoleSecurables(role).stream())
        .map(securable -> securable.getStringField(ApiManager.SECURABLE_ATTRIBUTE_NAME))
        .collect(Collectors.toList());

    allSecurables.addAll(getUserSecurables(username).stream()
        .map(securable -> securable.getStringField(ApiManager.SECURABLE_ATTRIBUTE_NAME))
        .collect(Collectors.toList()));

    allSecurables.addAll(getAuthenticatedSecurables().stream()
        .map(securable -> securable.getStringField(ApiManager.SECURABLE_ATTRIBUTE_NAME))
        .collect(Collectors.toList()));

    allSecurables.addAll(getPublicSecurables().stream()
        .map(securable -> securable.getStringField(ApiManager.SECURABLE_ATTRIBUTE_NAME))
        .collect(Collectors.toList()));

    if (allSecurables.isEmpty()) {
      // Fallback para roles não-enterprise
      allSecurables.addAll(listRoles);
    }

    return new HashSet<>(allSecurables);
  }

  public Set<GrantedAuthority> getAuthorities() {
    return this.getRoles().stream().map(SimpleGrantedAuthority::new).collect(Collectors.toSet());
  }

  public void updateTheme(String theme) throws Exception {
    Var userVar = this.getUserVar();
    if (userVar != null) {
      userVar.setField("theme", theme);
      try {
        DatabaseQueryManager authManager = new DatabaseQueryManager("auth");
        authManager.update(userVar);
      } catch (Exception e) {
        logger.error(e.getMessage());
      }
    }
  }

  public static void lockUser(User user) {
    if (user != null && user.getLockoutEnabled()) {
      user.setLockoutEnd(DateUtils.addMinutes(Date.from(Instant.now()), 10));
      try {
        DatabaseQueryManager authManager = new DatabaseQueryManager("auth");
        authManager.update(Var.valueOf(user));
      } catch (Exception e) {
        logger.error(e.getMessage(), e);
      }
    }
  }

  public static int getFailedAttempts(User user) {
    int failedAttempts = 0;
    if (user != null) {
      failedAttempts = user.getAccessFailedCount();
    }
    return failedAttempts;
  }

  public static boolean isUserLocked(User user) {
    if (user != null) {
      if (user.getLockoutEnd() != null) {
        return user.getLockoutEnd().after(Date.from(Instant.now()));
      }
    }

    return false;
  }

  public static void unlockUser(User user) {
    if (user != null && (user.getLockoutEnd() != null || user.getAccessFailedCount() > 0)) {
      user.setLockoutEnd(null);
      user.setAccessFailedCount(0);
      try {
        DatabaseQueryManager authManager = new DatabaseQueryManager("auth");
        authManager.update(Var.valueOf(user));
      } catch (Exception e) {
        logger.error(e.getMessage(), e);
      }
    }
  }

  public static void attemptFailed(User user) {
    try {
      if (user != null && user.getLockoutEnabled()) {
        int failedAttempts = user.getAccessFailedCount();
        user.setAccessFailedCount(failedAttempts + 1);
        DatabaseQueryManager authManager = new DatabaseQueryManager("auth");
        authManager.update(Var.valueOf(user));
      }
    } catch (Exception e) {
      logger.error(e.getMessage(), e);
    }
  }

  private Var getUserVar() throws Exception {
    return getUserVar(null);
  }

    public void setProviderInfo(String providerName, String providerKey) throws Exception {
    Var userVar = this.getUserVar();
    Var providerResult;

    if (userVar != null) {
      try {
        DatabaseQueryManager providerManager = new DatabaseQueryManager("userProviderKey");
        providerResult = providerManager.get(providerName, providerKey);
        if (providerManager.isDatabase()) {
          if (providerResult.size() > 0) {
            Var providerObject = providerResult.get(0);

            if (providerObject.getField("providerKey").getObjectAsString().equals(providerKey)) {
              return;
            }

            providerObject.setField("providerName", providerName);
            providerObject.setField("providerKey", providerKey);
            providerManager.insert(providerObject);
          } else {
            Map<String, Object> loginProviderObject = new LinkedHashMap<>();
            loginProviderObject.put("id", UUID.randomUUID());
            loginProviderObject.put("loginProvider", providerName);
            loginProviderObject.put("providerDisplayName", providerName);
            loginProviderObject.put("providerKey", providerKey);
            loginProviderObject.put("user", userVar);
            providerManager.insert(Var.valueOf(loginProviderObject));
          }
        }
      } catch (Exception e) {
        logger.error(e.getMessage());
      }
    }
  }

  public void updatePassword(String password) throws Exception {
    Var userVar = this.getUserVar();

    if (userVar != null) {
      userVar.setField("password", password);
      try {
        DatabaseQueryManager authManager = new DatabaseQueryManager("auth");
        authManager.update(userVar);
      } catch (Exception e) {
        logger.error(e.getMessage());
      }
    }
  }
}
