package cronapp.framework.rest;

import cronapi.AppConfig;
import cronapi.Var;
import cronapp.framework.CannotChangePasswordException;
import cronapp.framework.api.ApiManager;
import cronapp.framework.api.EventsManager;
import cronapp.framework.i18n.Messages;
import org.springframework.http.HttpStatus;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;
/**
 * Controller responsável por gerir a troca de
 * senha do usuário através de serviço REST
 *
 * @author Techne
 */
@RestController
@RequestMapping(value = "/changePassword")
public class ChangePassword {

  private final PasswordEncoder passwordEncoder = new BCryptPasswordEncoder();

  @ResponseStatus(HttpStatus.OK)
  @RequestMapping(method = RequestMethod.POST)
  public void post(String oldPassword, String newPassword, String newPasswordConfirmation) throws Exception {
    String authenticationType = AppConfig.type();

    if (authenticationType != null && !authenticationType.equalsIgnoreCase("internal") && !authenticationType.equalsIgnoreCase("token")) {
      throw new CannotChangePasswordException(Messages.getString("CannotChangePassword"));
    }

    if (!newPassword.equals(newPasswordConfirmation)) {
      throw new RuntimeException(Messages.getString("WrongConfirmationPassword"));
    }

    String username = AuthenticationUtil.getUsername();

    if (EventsManager.hasEvent("onChangePassword")) {
      EventsManager.executeEventOnTransaction("onChangePassword", Var.valueOf(username), Var.valueOf(oldPassword), Var.valueOf(newPassword), Var.valueOf(newPasswordConfirmation));
    } else {
      ApiManager apiManager = ApiManager.byUser(username);
      cronapp.framework.api.User user = apiManager.getUser();
      if (user == null) {
        throw new UsernameNotFoundException(Messages.getString("UserNotFound"));
      }
      String password = user.getPassword();
      if (!apiManager.passwordMatches(oldPassword, password)) {
        throw new BadCredentialsException(Messages.getString("UserOrPassordInvalids"));
      }
      String passwordEncoded = passwordEncoder.encode(newPassword);
      apiManager.updatePassword(passwordEncoded);
    }
  }
}