package cronapp.framework.authentication.token.google;

import cronapi.AppConfig;
import cronapp.framework.api.ApiManager;
import cronapp.framework.api.User;
import cronapp.framework.i18n.Messages;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.authentication.AuthenticationServiceException;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestOperations;
import org.springframework.web.client.RestTemplate;

import javax.servlet.http.HttpServletRequest;
import java.net.URI;
import java.util.Base64;
import java.util.Objects;
import java.util.regex.Pattern;

public class CaptchaVerify implements ICaptchaVerify {

  private static final Logger log = LoggerFactory.getLogger(CaptchaVerify.class);
  private static final String RECAPTCHA_TOKEN = "recaptchaToken";
  private static final Pattern RESPONSE_PATTERN = Pattern.compile("[A-Za-z0-9_-]+");

  private static final String URL_VERIFY = "https://www.google.com/recaptcha/api/siteverify?secret=%s&response=%s";
  private static String secretVerify;
  private static boolean userRecaptchaBackEnd;
  private RestOperations restTemplate;

  static {
    secretVerify = AppConfig.tokenRecaptcha();
    userRecaptchaBackEnd = StringUtils.hasText(secretVerify);
  }

  public CaptchaVerify() {
    this.restTemplate = new RestTemplate();
  }

  @Override
  public boolean processRequest(String username, HttpServletRequest request) throws Exception {
    if (!userRecaptchaBackEnd) {
      return true;
    }
    Assert.isTrue(request.getParameterMap().containsKey(RECAPTCHA_TOKEN), Messages.getString("TokenRecaptcha"));
    Assert.hasText(request.getParameter(RECAPTCHA_TOKEN), Messages.getString("TokenEmptyRecaptcha"));

    User user = ApiManager.byUser(username).getUser();

    verifyToken(user, request.getParameter(RECAPTCHA_TOKEN));

    final URI verifyUri = URI.create(String.format(getUrl(), getCaptchaSecret(), request.getParameter(RECAPTCHA_TOKEN)));
    try {
      final GoogleResponse googleResponse = restTemplate.getForObject(verifyUri, GoogleResponse.class);
      if (Objects.nonNull(googleResponse)) {
        String value = googleResponse.toString();
        log.debug("Google's response: {} ", value);
        if (!googleResponse.isSuccess()) {
          throw new AuthenticationServiceException(Messages.getString("TokenValidatedRecaptcha"));
        }
      }
    } catch (RestClientException rce) {
      throw new AuthenticationServiceException(Messages.getString("TokenRegistrationRecaptcha"), rce);
    }
    return true;
  }

  private void verifyToken(User user, String token) throws Exception {
    if (!responseSanityCheck(token)) {
      throw new ReCaptchaInvalidException(Messages.getString("RecaptchaInvalidCaracters"));
    }
  }

  private boolean responseSanityCheck(final String response) {
    return StringUtils.hasLength(response) && RESPONSE_PATTERN.matcher(response).matches();
  }

  @Override
  public String getCaptchaSecret() {
    return new String(Base64.getDecoder().decode(secretVerify)).trim();
  }

  @Override
  public String getUrl() {
    return URL_VERIFY;
  }
}
