package cronapp.framework.authentication.social.linkedin;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.gson.Gson;
import com.google.gson.JsonObject;
import cronapi.ErrorResponse;
import cronapp.framework.authentication.normal.AuthenticationConfigurer;
import cronapp.framework.authentication.social.SocialConfig;
import cronapp.framework.authentication.token.AuthenticationController;
import cronapp.framework.authentication.token.AuthenticationResponse;
import cronapp.framework.i18n.Messages;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.mobile.device.DeviceResolver;
import org.springframework.mobile.device.LiteDeviceResolver;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.net.URI;
import java.net.URLEncoder;
import java.util.Arrays;

@RestController
@RequestMapping(value = "/signin/linkedin")
public class LinkedinSignInController {
  private static final Logger logger = LoggerFactory.getLogger(LinkedinSignInController.class);
  private static final String LINKEDIN_OAUTH2_URL = "https://www.linkedin.com/oauth/v2/authorization";
  private static final String LINKEDIN_EXCHANGE_URL = "https://www.linkedin.com/oauth/v2/accessToken";
  private static final String LINKEDIN_PROFILE_URL = "https://api.linkedin.com/v2/me";
  private static final String LINKEDIN_EMAIL_ADDRESS_URL = "https://api.linkedin.com/v2/emailAddress?q=members&projection=(elements*(handle~))";
  @Autowired
  private HttpServletRequest request;
  @Autowired
  private HttpServletResponse response;
  @Autowired(required = false)
  private AuthenticationConfigurer authenticationConfigurer;
  @Autowired(required = false)
  private AuthenticationController authenticationController;

  @ExceptionHandler(Throwable.class)
  @ResponseBody
  public void handleControllerException(HttpServletRequest req, Throwable ex) throws IOException {
    logger.error("Linkedin Authentication Error", ex);
    ErrorResponse errorResponse = new ErrorResponse(HttpStatus.INTERNAL_SERVER_ERROR.value(), ex, req.getMethod());
    response.sendRedirect(request.getRequestURL().toString() + "/#/notconnected");
  }

  @ResponseStatus(HttpStatus.OK)
  @RequestMapping(method = RequestMethod.POST)
  public void post(@RequestParam(name = "scope", required = false) String scope) throws Exception {
    if (!SocialConfig.isEnabled("linkedin")) {
      logger.error("Login via Linkedin is disabled on your social config.");
      throw new Exception(Messages.getString("notAllowed"));
    }

    String clientId = SocialConfig.getProperties().getProperty("linkedin.appId");
    String redirectUri = URLEncoder.encode(request.getRequestURL().toString(), cronapi.CronapiConfigurator.ENCODING);
    String linkedinScope = "r_liteprofile%20r_emailaddress";

    response.sendRedirect(LINKEDIN_OAUTH2_URL +
        "?response_type=code" +
        "&client_id=" + clientId +
        "&redirect_uri=" + redirectUri +
        "&scope=" + linkedinScope
    );
  }

  @ResponseStatus(HttpStatus.OK)
  @RequestMapping(method = RequestMethod.GET)
  public void get(@RequestParam(name = "code", required = false) String code,
                  @RequestParam(name = "state", required = false) String state,
                  @RequestParam(name = "error", required = false) String error,
                  @RequestParam(name = "error_description", required = false) String errorDescription) throws Exception {
    if (!SocialConfig.isEnabled("linkedin")) {
      logger.error("Login via Linkedin is disabled on your social config.");
      throw new Exception(Messages.getString("notAllowed"));
    }
    if (!StringUtils.isBlank(error) || !StringUtils.isBlank(errorDescription)) {
      throw new Exception("An error has occurred: \n" + error + "\n" + errorDescription);
    }

    LinkedinAccessTokenTemplate accessCode = getLinkedinAccessToken(code);
    LinkedinProfileTemplate profile = getLinkedinProfile(accessCode.getAccessToken());

    JsonObject details = new JsonObject();
    details.addProperty("name", (profile.getFirstName() + " " + profile.getLastName()).trim());

    UsernamePasswordAuthenticationToken auth = new UsernamePasswordAuthenticationToken(profile.getEmailAddress(), "cronapp",
        Arrays.asList(new SimpleGrantedAuthority("#OAUTH#")));
    auth.setDetails(details);

    String redirect = request.getContextPath() + "/#/connected";

    if (authenticationConfigurer != null) {
      Authentication socialAuth = authenticationConfigurer.authenticate(auth);

      SecurityContextHolder.getContext().setAuthentication(socialAuth);

      try {
        request.getSession().setAttribute("#OAUTH#USER", profile.getEmailAddress());
      } catch (Exception e) {
        e.printStackTrace();
      }
    } else {
      DeviceResolver deviceResolver = new LiteDeviceResolver();

      ResponseEntity<AuthenticationResponse> authenticationRequest = authenticationController.auth(profile.getEmailAddress(), "cronapp",
          deviceResolver.resolveDevice(request), "cronapp", null, (JsonObject) new Gson().toJsonTree(profile), request);

      redirect = request.getContextPath() + "/#/connected?_ctk=" + authenticationRequest.getBody().getToken();
    }
    response.sendRedirect(redirect);
  }

  private LinkedinAccessTokenTemplate getLinkedinAccessToken(String authorizationCode) throws Exception {
    String clientId = SocialConfig.getProperties().getProperty("linkedin.appId");
    String clientSecret = SocialConfig.getProperties().getProperty("linkedin.appSecret");
    String redirectUri = URLEncoder.encode(request.getRequestURL().toString(), cronapi.CronapiConfigurator.ENCODING);

    String requestUrl = LINKEDIN_EXCHANGE_URL +
        "?grant_type=authorization_code" +
        "&code=" + authorizationCode +
        "&redirect_uri=" + redirectUri +
        "&client_id=" + clientId +
        "&client_secret=" + clientSecret;

    HttpPost httpPost = new HttpPost(requestUrl);
    CloseableHttpClient httpClient = HttpClients.createSystem();
    CloseableHttpResponse response = httpClient.execute(httpPost);
    String result = IOUtils.toString(response.getEntity().getContent(), "UTF-8");
    logger.info("getLinkedinAccessToken: " + result);
    LinkedinAccessTokenTemplate linkedinAccessToken = new ObjectMapper().readValue(result, LinkedinAccessTokenTemplate.class);
    return linkedinAccessToken;
  }

  private LinkedinProfileTemplate getLinkedinProfile(String accessToken) throws Exception {
    HttpGet httpGet = new HttpGet(LINKEDIN_PROFILE_URL);
    CloseableHttpClient httpClient = HttpClients.createSystem();
    httpGet.addHeader("Authorization", "Bearer " + accessToken);
    httpGet.addHeader("Connection", "Keep-Alive");
    CloseableHttpResponse responseProfile = httpClient.execute(httpGet);

    String result = IOUtils.toString(responseProfile.getEntity().getContent(), "UTF-8");
    logger.info("getLinkedinProfile: " + result);
    LinkedinProfileTemplate profile = new ObjectMapper().readValue(result, LinkedinProfileTemplate.class);
    // Add emailAddres

    httpGet.setURI(new URI(LINKEDIN_EMAIL_ADDRESS_URL));

    CloseableHttpResponse responseEmail = httpClient.execute(httpGet);
    String resultEmail = IOUtils.toString(responseEmail.getEntity().getContent(), "UTF-8");
    logger.info("getLinkedinProfileEmail: " + resultEmail);
    LinkedinProfileTemplate tmpProfile = new ObjectMapper().readValue(resultEmail, LinkedinProfileTemplate.class);
    profile.setEmailAddress(tmpProfile.getEmailAddress());
    return profile;
  }
}

