package cronapp.framework.authentication.sso;

import cronapi.Var;
import cronapp.framework.api.ApiManager;
import cronapp.framework.api.EventsManager;
import cronapp.framework.authentication.AuthenticationUtil;
import cronapp.framework.authentication.security.CronappAnonymousAuthenticationFilter;
import cronapp.framework.authentication.security.FilterInvocationVoter;
import cronapp.framework.authentication.security.Permission;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.security.oauth2.OAuth2AutoConfiguration;
import org.springframework.boot.autoconfigure.security.oauth2.client.OAuth2SsoProperties;
import org.springframework.boot.autoconfigure.security.oauth2.resource.ResourceServerTokenServicesConfiguration;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.security.access.AccessDecisionVoter;
import org.springframework.security.access.vote.UnanimousBased;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter;
import org.springframework.security.oauth2.client.OAuth2ClientContext;
import org.springframework.security.oauth2.client.OAuth2RestTemplate;
import org.springframework.security.oauth2.client.resource.OAuth2ProtectedResourceDetails;
import org.springframework.security.oauth2.config.annotation.web.configuration.EnableOAuth2Client;
import org.springframework.security.oauth2.config.annotation.web.configuration.OAuth2ClientConfiguration;
import org.springframework.security.web.authentication.AnonymousAuthenticationFilter;
import org.springframework.security.web.authentication.Http403ForbiddenEntryPoint;
import org.springframework.security.web.authentication.logout.LogoutSuccessHandler;

import javax.servlet.http.HttpServletResponse;
import java.util.List;

@Configuration
@EnableOAuth2Client
@AutoConfigureBefore({OAuth2AutoConfiguration.class})
@EnableConfigurationProperties(OAuth2SsoProperties.class)
@Import({OAuth2ClientConfiguration.class, ResourceServerTokenServicesConfiguration.class})
public class AuthorizationConfigurer extends WebSecurityConfigurerAdapter {

    @Autowired(required = false)
    private Permission permission;

    @Autowired
    private ApplicationContext applicationContext;

    @Autowired
    private List<AccessDecisionVoter<? extends Object>> decisionVoters;

    @Override
    protected void configure(HttpSecurity http) throws Exception {
        // post sem csrf
        http.csrf().disable();

        // security permission
        if (permission == null) {
            http.anonymous().authenticationFilter(new CronappAnonymousAuthenticationFilter("anonymousAuthenticationFilterKey", "anonymousUser", ApiManager.getPublicAuthorities()));
            http.authorizeRequests().anyRequest().denyAll().accessDecisionManager(new UnanimousBased(decisionVoters));
        } else {
            permission.loadSecurityPermission(http);
            AuthenticationUtil.loadStaticSecurity(http);
        }

        // logout
        http.exceptionHandling()
            .authenticationEntryPoint(new Http403ForbiddenEntryPoint())
            .and()
            .logout()
            .logoutSuccessHandler(logoutHandler())
            .deleteCookies("_u");

        // x-frame-options disable
        http.headers().cacheControl().disable().frameOptions().disable().httpStrictTransportSecurity().disable();

        new SsoSecurityConfigurer(applicationContext).configure(http);
    }

    @Bean
    public OAuth2RestTemplate oauth2RestTemplate(OAuth2ClientContext oauth2ClientContext, OAuth2ProtectedResourceDetails details) {
        return new OAuth2RestTemplate(details, oauth2ClientContext);
    }

    @Bean
    public AccessDecisionVoter filterInvocationVoter() {
        return new FilterInvocationVoter();
    }

    private LogoutSuccessHandler logoutHandler() {
        return (request, response, authentication) -> {
            if (EventsManager.hasEvent("onLogout") && authentication != null && authentication.getName() != null) {
                EventsManager.executeEventOnTransaction("onLogout", Var.valueOf(authentication.getName()));
            }

            if (request.getHeader("Accept") == null || !request.getHeader("Accept").contains("json")) {
                response.setStatus(HttpServletResponse.SC_OK);
                response.sendRedirect("/index.html");
            }
        };
    }
}