package cronapp.framework.authentication.token;

import com.google.gson.JsonObject;
import cronapi.AppConfig;
import cronapi.RestClient;
import cronapi.Var;
import cronapp.framework.LockedUserException;
import cronapp.framework.api.ApiManager;
import cronapp.framework.api.EventsManager;
import cronapp.framework.api.User;
import cronapp.framework.authentication.external.ExternalAuthenticationConfig;
import cronapp.framework.authentication.security.CronappUserDetails;
import cronapp.framework.authentication.social.SocialConfig;
import cronapp.framework.authentication.token.google.CaptchaVerify;
import cronapp.framework.authentication.token.google.ICaptchaVerify;
import cronapp.framework.i18n.Messages;
import cronapp.framework.tenant.TenantComponent;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.ResponseEntity;
import org.springframework.lang.Nullable;
import org.springframework.mobile.device.Device;
import org.springframework.security.authentication.AuthenticationServiceException;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.util.Assert;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import java.util.*;

import static cronapp.framework.authentication.external.ExternalAuthenticationConfig.*;

@RestController
@RequestMapping("auth")
public class AuthenticationController {

  private static final Logger log = LoggerFactory.getLogger(AuthenticationController.class);

  private TenantComponent tenantComponent;

  private ICaptchaVerify captchaVerify;

  public AuthenticationController(@Nullable TenantComponent tenantComponent) {
    this.captchaVerify = new CaptchaVerify();
    this.tenantComponent = tenantComponent;
  }

  @RequestMapping(method = RequestMethod.POST)
  public ResponseEntity<AuthenticationResponse> authenticationRequest(@RequestParam String username, String password, Device device, @RequestHeader(name = "X-AUTH-TOKEN", required = false) String token, final HttpServletRequest request) throws AuthenticationException {
    return auth(username, password, device, "local", token, null, request);
  }

  public ResponseEntity<AuthenticationResponse> auth(String username, String password, Device device, String provider, String authToken, JsonObject details, final HttpServletRequest request) throws AuthenticationException {

    if (authToken != null) {
      String localProvider = TokenUtils.getProviderFromToken(authToken);
      if (provider != null && !"local".equals(localProvider) && username.equals("#OAUTH#")) {
        username = TokenUtils.getUsernameFromToken(authToken);
        provider = localProvider;
      }
    }

    boolean isOauth = !"local".equals(provider);
    boolean autoSignUp = false;
    boolean externalAuthenticationSuccess = false;

    Authentication externalToken;
    CronappUserDetails externalUserDetails = null;

    if (isExternalAuth()) {
      externalToken = authenticateExternally(new UsernamePasswordAuthenticationToken(username, password));
      provider = getExternalAuthType();
      externalAuthenticationSuccess = true;
      autoSignUp = AppConfig.autoSignUp();

      if (externalToken.getPrincipal() instanceof CronappUserDetails) {
        externalUserDetails = (CronappUserDetails) externalToken.getPrincipal();
      }
    } else if (ExternalAuthenticationConfig.isSocial(provider)) {
      autoSignUp = SocialConfig.isAutoSignUp();
    }

    ApiManager apiManager = ApiManager.byUserAndPassword(username, password, provider, autoSignUp, details);

    User userNoPassword = null;
    try {

      userNoPassword = ApiManager.byUser(username).getUser();

      if (ApiManager.isUserLocked(userNoPassword)) {
        throw new LockedUserException(Messages.getString("UserLocked"));
      }

      verifyRecaptcha(username, request);

      User user = apiManager.getUser(externalUserDetails);

      if (user == null) {
        throw new UsernameNotFoundException(Messages.getString("UserNotFound"));
      }

      if (isExternalAuth() && (!isOauth && !externalAuthenticationSuccess)) {
        throw new BadCredentialsException(Messages.getString("UserOrPassordInvalids"));
      }

      if (!isExternalAuth() && (!isOauth && !apiManager.passwordMatches(password, user.getPassword()))) {
        throw new BadCredentialsException(Messages.getString("UserOrPassordInvalids"));
      }

      Collection<? extends GrantedAuthority> authorities;

      if (externalUserDetails == null) {
        authorities = apiManager.getAuthorities();
      } else {
        authorities = externalUserDetails.getAuthorities();
      }

      UserDetails userDetails = new org.springframework.security.core.userdetails.User(username, "password", true, true, true, true, authorities);

      SecurityContextHolder.getContext()
          .setAuthentication(new UsernamePasswordAuthenticationToken(userDetails, "password", authorities));

      StringJoiner roles = new StringJoiner(",");

      roles.add("Public");
      roles.add("Authenticated");

      boolean root = false;
      for (GrantedAuthority role : authorities) {
        roles.add(role.getAuthority());
        if (role.getAuthority().equalsIgnoreCase("Administrators")) {
          root = true;
        }
      }

      if (tenantComponent != null) {
        tenantComponent.authenticationTenant(user.getUsername());
      }

      User userWithoutPass = user.resetPassword();

      try {
        // Only the server events will be fired here
        if (EventsManager.hasEvent("onLogin") && EventsManager.getEvent("onLogin").get("type").getAsString().equalsIgnoreCase("server")) {
          EventsManager.executeEventOnTransaction("onLogin", Var.valueOf(username));
        }
      } catch (Exception e) {
        log.error("It was not possible to execute the 'onLogin' event. It may be malformed. Please, try to attach the event again.", e);
      }

      String token = TokenUtils.generateToken(userDetails, user.getName(), device, provider);
      Date expires = TokenUtils.getExpirationDateFromToken(token);

      ApiManager.unlockUser(user);
      return ResponseEntity.ok(new AuthenticationResponse(userWithoutPass, token, expires.getTime(), roles.toString(), root));
    } catch (Exception e) {
      log.error(Messages.getString("AuthError", e.getMessage()), e);
      ApiManager.attemptFailed(userNoPassword);

      if (!ApiManager.isUserLocked(userNoPassword) && ApiManager.getFailedAttempts(userNoPassword) > 4) {
        ApiManager.lockUser(userNoPassword);
        throw new AuthenticationServiceException(Messages.getString("UserLocked10Min"));
      }

      throw new AuthenticationServiceException(Messages.getString("AuthError", e.getMessage()));
    }
  }

  @RequestMapping(value = "refresh", method = RequestMethod.GET)
  public ResponseEntity<?> authenticationRequest(HttpServletRequest request) {
    String tokenHeader = TokenUtils.AUTH_HEADER_NAME;
    String token = request.getHeader(tokenHeader);
    Date expires = TokenUtils.getExpirationDateFromToken(token);

    if (!TokenUtils.canTokenBeRefreshed(token, expires)) {
      return ResponseEntity.badRequest().body(null);
    }

    String refreshedToken = TokenUtils.refreshToken(token);
    expires = TokenUtils.getExpirationDateFromToken(token);
    String username = TokenUtils.getUsernameFromToken(token);
    String name = TokenUtils.getNameFromToken(token);
    if ((name == null) || ("".equals(name))) {
      name = username;
    }

    ApiManager apiManager = ApiManager.byUser(username);

    StringJoiner roles = new StringJoiner(",");

    roles.add("Public");
    roles.add("Authenticated");

    boolean root = false;
    for (GrantedAuthority role : apiManager.getAuthorities()) {
      roles.add(role.getAuthority());
      if (role.getAuthority().equalsIgnoreCase("Administrators")) {
        root = true;
      }
    }

    User userWithoutPass = new User(name, username);

    AuthenticationResponse authenticationResponse = new AuthenticationResponse(userWithoutPass, refreshedToken,
        expires.getTime(), roles.toString(), root);

    return ResponseEntity.ok(authenticationResponse);
  }

  private void verifyRecaptcha(String username, HttpServletRequest request) {
    try {
      Assert.isTrue(captchaVerify.processRequest(username, request), "");
    } catch (Exception e) {
      log.error(Messages.getString("AuthError", e.getMessage()), e);
      throw new AuthenticationServiceException(Messages.getString("AuthError", e.getMessage()));
    }
  }
}
