package cronapp.framework.authentication.social;

import com.google.gson.JsonObject;
import cronapp.framework.authentication.normal.AuthenticationConfigurer;
import cronapp.framework.authentication.token.AuthenticationController;
import cronapp.framework.authentication.token.AuthenticationResponse;
import org.springframework.http.ResponseEntity;
import org.springframework.lang.Nullable;
import org.springframework.mobile.device.DeviceResolver;
import org.springframework.mobile.device.LiteDeviceResolver;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.social.connect.Connection;
import org.springframework.social.connect.web.SignInAdapter;
import org.springframework.stereotype.Service;
import org.springframework.web.context.request.NativeWebRequest;

import javax.servlet.http.HttpServletRequest;
import java.util.Collections;

@Service
public class SocialSignInAdapter implements SignInAdapter {

  private AuthenticationConfigurer authenticationConfigurer;

  private AuthenticationController authenticationController;

  private HttpServletRequest servletRequest;

  public SocialSignInAdapter(HttpServletRequest servletRequest,
                             @Nullable AuthenticationConfigurer authenticationConfigurer,
                             @Nullable AuthenticationController authenticationController) {
    this.authenticationConfigurer = authenticationConfigurer;
    this.authenticationController = authenticationController;
    this.servletRequest = servletRequest;
  }

  @Override
  public String signIn(String localUserId, Connection<?> connection, NativeWebRequest request) {

    try {
      String email = connection.fetchUserProfile().getEmail();

      if (email == null) {
        email = connection.fetchUserProfile().getUsername();
      }

      JsonObject json = new JsonObject();
      json.addProperty("name", connection.getDisplayName());
      json.addProperty("image", connection.getImageUrl());

      String provider = connection.getKey().getProviderId();

      UsernamePasswordAuthenticationToken auth = new UsernamePasswordAuthenticationToken(
          email, provider, Collections.singletonList(new SimpleGrantedAuthority("#OAUTH#")));

      auth.setDetails(json);

      try {
        if (authenticationConfigurer != null) {
          Authentication socialAuth = authenticationConfigurer.authenticate(auth);

          SecurityContextHolder.getContext().setAuthentication(socialAuth);

          servletRequest.getSession().setAttribute("#OAUTH#USER", email);
        } else {
          DeviceResolver deviceResolver = new LiteDeviceResolver();

          ResponseEntity<AuthenticationResponse> authenticationRequest = authenticationController.auth(email, provider, deviceResolver.resolveDevice(servletRequest), provider, null, json, servletRequest);

          return servletRequest.getContextPath() + "/#/connected?_ctk=" + authenticationRequest.getBody().getToken();
        }
      } catch (Exception e) {
        return servletRequest.getContextPath() + "/#/notconnected";
      }
    } catch (Exception e) {
      return servletRequest.getContextPath() + "/#/error/403";
    }

    return servletRequest.getContextPath() + "/#/connected";

  }
}
