package cronapp.framework.api;

import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.concurrent.Callable;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;

import cronapi.ErrorResponse;
import cronapi.Var;
import cronapi.database.TransactionManager;
import cronapi.util.Operations;
import org.springframework.http.HttpStatus;

public class EventsManager {
  private static JsonObject JSON;
  
  static {
    JSON = loadJSON();
  }
  
  private static JsonObject loadJSON() {
    ClassLoader classLoader = EventsManager.class.getClassLoader();
    try (InputStream stream = classLoader.getResourceAsStream("META-INF/events.json")) {
      InputStreamReader reader = new InputStreamReader(stream);
      JsonElement jsonElement = new JsonParser().parse(reader);
      return jsonElement.getAsJsonObject();
    }
    catch(Exception e) {
      return new JsonObject();
    }
  }
  
  private static JsonObject getJSON() {
    if(Operations.IS_DEBUG) {
      return loadJSON();
    }
    else {
      return JSON;
    }
  }
  
  public static JsonObject getEvent(String eventName) {
    JsonObject obj = getJSON().getAsJsonObject(eventName);
    return obj;
  }
  
  private static boolean isNull(JsonElement value) {
    return value == null || value.isJsonNull();
  }

  public static Var executeEvent(String eventName, Var... params) {
    JsonObject event = getEvent(eventName);
    if(event != null) {
        return executeEvent(event, params);
    }

    return null;
  }

  public static Var executeEventOnTransaction(String eventName, Var... params) {
    return runIntoTransaction(() -> {
      JsonObject event = getEvent(eventName);
      if (event != null) {
        return executeEvent(event, params);
      }

      return null;
    });
  }

  private static Var runIntoTransaction(Callable<Var> callable) {
    Var var = Var.VAR_NULL;
    try {
      var = callable.call();
      TransactionManager.commit();
    }
    catch(Exception ex) {
      TransactionManager.rollback();
      ErrorResponse errorResponse = new ErrorResponse(HttpStatus.INTERNAL_SERVER_ERROR.value(), ex, "GET");
      throw new RuntimeException(errorResponse.getError(), ex);
    }
    return var;
  }
  
  public static boolean hasEvent(String eventName) {
    return !isNull(getJSON().get(eventName));
  }
  
  public static Var executeEvent(JsonObject event, Var... params) {
    if(event != null) {
      Var name = Var.valueOf(event.get("blocklyClass").getAsString() + ":" + event.get("blocklyMethod").getAsString());
      try {
        return Operations.callBlockly(name, params);
      }
      catch(Exception e) {
        throw new RuntimeException(e);
      }
    }

    return null;
  }
}
