package cronapp.framework.authentication.saml;

import cronapp.framework.api.ApiManager;
import cronapp.framework.authentication.security.CronappUserDetails;
import org.opensaml.xml.schema.impl.XSStringImpl;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.security.saml.SAMLCredential;
import org.springframework.security.saml.userdetails.SAMLUserDetailsService;

import java.time.OffsetDateTime;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

public class SamlUserDetailsService implements SAMLUserDetailsService {
  @Override
  public Object loadUserBySAML(SAMLCredential credential) throws UsernameNotFoundException {

    String userName = credential.getNameID().getValue();
    String normalizedUserName = ApiManager.normalize(userName);
    String email = null;

    if (email == null) {
      email = normalizedUserName + "@no-email";
    }

    String normalizedEmail = ApiManager.normalize(email);

    Set<String> securables = credential.getAttributes().stream()
        .filter(attribute -> attribute.getName().equals("Role"))
        .flatMap(attribute -> attribute.getAttributeValues().stream())
        .filter(attribute -> attribute instanceof XSStringImpl)
        .map(attribute -> (XSStringImpl) attribute)
        .map(XSStringImpl::getValue)
        .flatMap(authority -> ApiManager.getRoleSecurables(authority).stream())
        .map(securable -> securable.getStringField(ApiManager.SECURABLE_ATTRIBUTE_NAME))
        .collect(Collectors.toSet());

    securables.addAll(ApiManager.getUserSecurables(userName).stream()
        .map(securable -> securable.getStringField(ApiManager.SECURABLE_ATTRIBUTE_NAME))
        .collect(Collectors.toList()));

    securables.addAll(ApiManager.getAuthenticatedSecurables().stream()
        .map(securable -> securable.getStringField(ApiManager.SECURABLE_ATTRIBUTE_NAME))
        .collect(Collectors.toList()));

    securables.addAll(ApiManager.getPublicSecurables().stream()
        .map(securable -> securable.getStringField(ApiManager.SECURABLE_ATTRIBUTE_NAME))
        .collect(Collectors.toList()));

    Set<GrantedAuthority> mappedAuthorities = securables.stream().distinct()
        .sorted()
        .map(SimpleGrantedAuthority::new)
        .collect(Collectors.toSet());

    return CronappUserDetails.newBuilder()
        .setName(userName)
        .setUserName(userName)
        .setNormalizedUserName(normalizedUserName)
        .setEmail(email)
        .setNormalizedEmail(normalizedEmail)
        .setEmailConfirmed(true)
        .setSecurityStamp(UUID.randomUUID().toString())
        .setPhoneNumberConfirmed(true)
        .setTwoFactorEnabled(false)
        .setLockoutEnd(OffsetDateTime.MIN)
        .setLockoutEnabled(false)
        .setAccessFailedCount(0)
        .setAuthorities(mappedAuthorities)
        .build();
  }
}
