package cronapp.framework.authentication.token;

import cronapi.AppConfig;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SignatureAlgorithm;
import org.springframework.mobile.device.Device;
import org.springframework.security.core.userdetails.UserDetails;

import java.util.Date;
import java.util.HashMap;
import java.util.Map;

public class TokenUtils {
  
  private static String secret;

  private static final String AUDIENCE_MOBILE = "mobile";

  private static final String AUDIENCE_TABLET = "tablet";
  
  public static final String AUTH_HEADER_NAME = "X-AUTH-TOKEN";

  static {
    secret = AppConfig.token();
  }

  public static String getUsernameFromToken(String token) {
    String username;
    try {
      Claims claims = getClaimsFromToken(token);
      username = claims.getSubject();
    }
    catch(Exception e) {
      username = null;
    }
    return username;
  }

  public static String getProviderFromToken(String token) {
    String provider;
    try {
      Claims claims = getClaimsFromToken(token);
      provider = claims.getIssuer();
    }
    catch(Exception e) {
      provider = null;
    }
    return provider;
  }
  
  private static Date getCreatedDateFromToken(String token) {
    Date created;
    try {
      Claims claims = getClaimsFromToken(token);
      created = new Date((Long)claims.get("created"));
    }
    catch(Exception e) {
      created = null;
    }
    return created;
  }
  
  static Date getExpirationDateFromToken(String token) {
    Date expiration;
    try {
      Claims claims = getClaimsFromToken(token);
      expiration = claims.getExpiration();
    }
    catch(Exception e) {
      expiration = null;
    }
    return expiration;
  }
  
  private static String getAudienceFromToken(String token) {
    String audience;
    try {
      Claims claims = getClaimsFromToken(token);
      audience = (String)claims.get("audience");
    }
    catch(Exception e) {
      audience = null;
    }
    return audience;
  }
  
  private static Claims getClaimsFromToken(String token) {
    Claims claims;
    try {
      claims = Jwts.parser().setSigningKey(secret).parseClaimsJws(token).getBody();
    }
    catch(Exception e) {
      claims = null;
    }
    return claims;
  }

  public static String getNameFromToken(String token) {
    String name;
    try {
      Claims claims = getClaimsFromToken(token);
      name = (String)claims.get("name");
    }
    catch(Exception e) {
      name = null;
    }
    return name;
  }

  private static Date generateCurrentDate() {
    return new Date(System.currentTimeMillis());
  }
  
  private static Date generateExpirationDate() {
    Long expiration = AppConfig.tokenExpiration();
    return new Date(System.currentTimeMillis() + expiration * 1000);
  }
  
  public static boolean isTokenExpired(String token) {
    Date expiration = getExpirationDateFromToken(token);
    return expiration.before(generateCurrentDate());
  }
  
  private static boolean isCreatedBeforeLastPasswordReset(Date created, Date lastPasswordReset) {
    return (lastPasswordReset != null && created.before(lastPasswordReset));
  }
  
  private static String generateAudience(Device device) {
    String audience = "unknown";
    if(device.isNormal()) {
      audience = "web";
    }
    else if(device.isTablet()) {
      audience = AUDIENCE_TABLET;
    }
    else if(device.isMobile()) {
      audience = AUDIENCE_MOBILE;
    }
    return audience;
  }
  
  private static boolean ignoreTokenExpiration(String token) {
    String audience = getAudienceFromToken(token);
    return (AUDIENCE_TABLET.equals(audience) || AUDIENCE_MOBILE.equals(audience));
  }
  
  static String generateToken(UserDetails userDetails, Device device) {
    Map<String, Object> claims = new HashMap<>();
    claims.put("sub", userDetails.getUsername());
    claims.put("audience", generateAudience(device));
    claims.put("created", generateCurrentDate());
    claims.put("iss", "local");
    return generateToken(claims);
  }

  static String generateToken(UserDetails userDetails, Device device, String provider) {
    Map<String, Object> claims = new HashMap<>();
    claims.put("sub", userDetails.getUsername());
    claims.put("audience", generateAudience(device));
    claims.put("created", generateCurrentDate());
    claims.put("iss", provider);
    return generateToken(claims);
  }

  static String generateToken(UserDetails userDetails, String name, Device device, String provider) {
    Map<String, Object> claims = new HashMap<>();
    claims.put("sub", userDetails.getUsername());
    claims.put("audience", generateAudience(device));
    claims.put("created", generateCurrentDate());
    claims.put("iss", provider);
    claims.put("name", name);
    return generateToken(claims);
  }
  
  private static String generateToken(Map<String, Object> claims) {
    return Jwts.builder().setClaims(claims).setExpiration(generateExpirationDate())
            .signWith(SignatureAlgorithm.HS512, secret).compact();
  }
  
  static boolean canTokenBeRefreshed(String token, Date lastPasswordReset) {
    Date created = getCreatedDateFromToken(token);
    return ((isCreatedBeforeLastPasswordReset(created, lastPasswordReset)) &&
            (!(isTokenExpired(token)) || ignoreTokenExpiration(token)));
  }
  
  static String refreshToken(String token) {
    String refreshedToken;
    try {
      Claims claims = getClaimsFromToken(token);
      claims.put("created", generateCurrentDate());
      refreshedToken = generateToken(claims);
    }
    catch(Exception e) {
      refreshedToken = null;
    }
    return refreshedToken;
  }
}
