package cronapp.framework.authentication.sso;

import com.google.gson.Gson;
import com.google.gson.JsonObject;
import cronapi.Var;
import cronapp.framework.api.ApiManager;
import cronapp.framework.api.EventsManager;
import cronapp.framework.i18n.Messages;
import org.springframework.boot.autoconfigure.security.oauth2.client.OAuth2SsoProperties;
import org.springframework.boot.autoconfigure.security.oauth2.resource.UserInfoRestTemplateFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.security.authentication.AuthenticationServiceException;
import org.springframework.security.config.annotation.SecurityConfigurerAdapter;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configurers.ExceptionHandlingConfigurer;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.security.oauth2.client.OAuth2RestOperations;
import org.springframework.security.oauth2.client.filter.OAuth2ClientAuthenticationProcessingFilter;
import org.springframework.security.oauth2.provider.token.ResourceServerTokenServices;
import org.springframework.security.web.DefaultSecurityFilterChain;
import org.springframework.security.web.authentication.AuthenticationFailureHandler;
import org.springframework.security.web.authentication.AuthenticationSuccessHandler;
import org.springframework.security.web.authentication.HttpStatusEntryPoint;
import org.springframework.security.web.authentication.LoginUrlAuthenticationEntryPoint;
import org.springframework.security.web.authentication.preauth.AbstractPreAuthenticatedProcessingFilter;
import org.springframework.security.web.authentication.session.SessionAuthenticationStrategy;
import org.springframework.security.web.util.matcher.MediaTypeRequestMatcher;
import org.springframework.security.web.util.matcher.RequestHeaderRequestMatcher;
import org.springframework.web.accept.ContentNegotiationStrategy;
import org.springframework.web.accept.HeaderContentNegotiationStrategy;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletResponse;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.StringJoiner;

class SsoSecurityConfigurer {

    private final ApplicationContext applicationContext;

    SsoSecurityConfigurer(ApplicationContext applicationContext) {
        this.applicationContext = applicationContext;
    }

    public void configure(HttpSecurity http) throws Exception {
        OAuth2SsoProperties sso = this.applicationContext.getBean(OAuth2SsoProperties.class);
        // Delay the processing of the filter until we know the SessionAuthenticationStrategy is available:
        http.apply(new OAuth2ClientAuthenticationConfigurer(oauth2SsoFilter(sso)));
        addAuthenticationEntryPoint(http, sso);
    }

    private void addAuthenticationEntryPoint(HttpSecurity http, OAuth2SsoProperties sso) throws Exception {
        ExceptionHandlingConfigurer<HttpSecurity> exceptions = http.exceptionHandling();
        ContentNegotiationStrategy contentNegotiationStrategy = http.getSharedObject(ContentNegotiationStrategy.class);
        if (contentNegotiationStrategy == null) {
            contentNegotiationStrategy = new HeaderContentNegotiationStrategy();
        }
        MediaTypeRequestMatcher preferredMatcher = new MediaTypeRequestMatcher(
                contentNegotiationStrategy, MediaType.APPLICATION_XHTML_XML,
                new MediaType("image", "*"), MediaType.TEXT_HTML, MediaType.TEXT_PLAIN);
        preferredMatcher.setIgnoredMediaTypes(Collections.singleton(MediaType.ALL));
        exceptions.defaultAuthenticationEntryPointFor(
                new LoginUrlAuthenticationEntryPoint(sso.getLoginPath()),
                preferredMatcher);
        // When multiple entry points are provided the default is the first one
        exceptions.defaultAuthenticationEntryPointFor(
                new HttpStatusEntryPoint(HttpStatus.UNAUTHORIZED),
                new RequestHeaderRequestMatcher("X-Requested-With", "XMLHttpRequest"));
    }

    private OAuth2ClientAuthenticationProcessingFilter oauth2SsoFilter(OAuth2SsoProperties sso) {
        OAuth2RestOperations restTemplate = this.applicationContext.getBean(UserInfoRestTemplateFactory.class).getUserInfoRestTemplate();
        ResourceServerTokenServices tokenServices = this.applicationContext.getBean(ResourceServerTokenServices.class);
        OAuth2ClientAuthenticationProcessingFilter filter = new OAuth2ClientAuthenticationProcessingFilter(sso.getLoginPath());
        filter.setRestTemplate(restTemplate);
        filter.setTokenServices(tokenServices);
        filter.setApplicationEventPublisher(this.applicationContext);
        filter.setAuthenticationSuccessHandler(successHandler());
        filter.setAuthenticationFailureHandler(failureHandler());
        return filter;
    }

    /**
     * Handler para sucesso de autorização
     */
    private AuthenticationSuccessHandler successHandler() {
        return (request, response, authentication) -> {
            String name = authentication.getName();
            String rawPassword = authentication.getCredentials().toString();

            ApiManager apiManager = ApiManager.byUserAndPassword(name, rawPassword, "openid", true, null);

            try {
                cronapp.framework.api.User user = apiManager.getUser();

                if (user == null) {
                    throw new UsernameNotFoundException(Messages.getString("UserNotFound"));
                }

                StringJoiner roles = new StringJoiner(",");
                roles.add("Public");
                roles.add("Authenticated");

                boolean root = false;
                for (GrantedAuthority role : authentication.getAuthorities()) {
                    roles.add(role.getAuthority());
                    if (role.getAuthority().equalsIgnoreCase("Administrators")) {
                        root = true;
                    }
                }

                Gson gson = new Gson();
                JsonObject json = new JsonObject();
                json.add("user", gson.toJsonTree(user.resetPassword()));
                json.add("picture", gson.toJsonTree(user.getPicture()));
                json.addProperty("roles", roles.toString());
                json.addProperty("theme", "");
                json.addProperty("root", root);

                response.addCookie(new Cookie("_u", URLEncoder.encode(json.toString(), StandardCharsets.UTF_8.name())));

                response.setStatus(HttpServletResponse.SC_OK);

                response.sendRedirect("/");
            } catch (Exception e) {
                throw new AuthenticationServiceException(Messages.getString("AuthError", e.getMessage()));
            }

            if (EventsManager.hasEvent("onLogin")) {
                EventsManager.executeEventOnTransaction("onLogin", Var.valueOf(authentication.getName()));
            }
        };
    }

    /**
     * Handler para falha de autorização
     */
    private AuthenticationFailureHandler failureHandler() {
        return (request, response, failureHandler) -> response.setStatus(HttpStatus.UNAUTHORIZED.value());
    }

    private static class OAuth2ClientAuthenticationConfigurer
            extends SecurityConfigurerAdapter<DefaultSecurityFilterChain, HttpSecurity> {

        private OAuth2ClientAuthenticationProcessingFilter filter;

        OAuth2ClientAuthenticationConfigurer(OAuth2ClientAuthenticationProcessingFilter filter) {
            this.filter = filter;
        }

        @Override
        public void configure(HttpSecurity builder) {
            OAuth2ClientAuthenticationProcessingFilter ssoFilter = this.filter;
            ssoFilter.setSessionAuthenticationStrategy(builder.getSharedObject(SessionAuthenticationStrategy.class));
            builder.addFilterAfter(ssoFilter, AbstractPreAuthenticatedProcessingFilter.class);
        }
    }
}