package cronapp.framework.api;

import java.util.*;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;
import org.springframework.security.crypto.password.PasswordEncoder;

import cronapi.Var;
import cronapi.database.DataSource;
import cronapi.database.DatabaseQueryManager;
import cronapp.framework.authentication.normal.AuthenticationConfigurer;
import com.google.gson.JsonObject;

public final class ApiManager {

  private static final Logger log = LoggerFactory.getLogger(AuthenticationConfigurer.class);

  private final String username;

  private final String password;

  private final String type;

  private final boolean autoSignUp;

  private Boolean passwordMatches = false;

  private final PasswordEncoder passwordEncoder = new BCryptPasswordEncoder();
  
  private JsonObject details;

  private ApiManager(String username, String password, String type, boolean autoSignUp, JsonObject details) {
    this.username = username;
    this.password = password;
    this.type = (type==null?"local":type);
    this.autoSignUp = autoSignUp;
    this.details = details;
    try {
      DatabaseQueryManager authManager = new DatabaseQueryManager("auth");
      passwordMatches = !EventsManager.hasEvent("onAuthenticate") && authManager.isDatabase();
    } catch(Exception e) {
      //Abafa
    }
  }

  public boolean passwordMatches(CharSequence rawPassword, String password) {
    if(passwordMatches) {
      return passwordEncoder.matches(rawPassword, password);
    }
    else {
      return true;
    }
  }

  public static ApiManager byUser(String username) {
    return new ApiManager(username, null, "local", false, null);
  }

  public static ApiManager byUserAndPassword(String username, String password, String type, boolean autoSignUp, JsonObject details) {
    return new ApiManager(username, password, type, autoSignUp, details);
  }

  public static ApiManager byUserAndPassword(String username, String password) {
    return new ApiManager(username, password, "local", false, null);
  }
  
  private String getDetail(String key, String defValue) {
    if (details != null && details.get(key) != null && !details.get(key).getAsString().isEmpty()) {
      return details.get(key).getAsString();
    }
    
    return defValue;
  }

  private Var getUserVar() throws Exception {
    Var userResult = null;
    if (EventsManager.hasEvent("onAuthenticate")) {
      userResult = EventsManager.executeEventOnTransaction("onAuthenticate", Var.valueOf(username), Var.valueOf(password), Var.valueOf(type));
    } else {
      DatabaseQueryManager authManager = new DatabaseQueryManager("auth");
      userResult = authManager.get(username);

      if(userResult.getType() == Var.Type.LIST && userResult.size() == 0 &&  autoSignUp && !("local".equals(type))) {
        Map<String, String> map = new LinkedHashMap<>();
        map.put("name", getDetail("name", username));
        map.put("login", getDetail("login", username));
        map.put("email", getDetail("email", username));
      
        map.put("image", getDetail("image", null));
        map.put("picture", getDetail("image", null));
          
        map.put("password", UUID.randomUUID().toString());
        map.put("theme", getDetail("theme", ""));
        
        userResult = Var.valueOf(map);
        authManager.insert(userResult);
      }

    }
    if(userResult.getType() == Var.Type.BOOLEAN) {
      if(userResult.getObjectAsBoolean()) {
        Map<String, String> map = new LinkedHashMap<>();
        map.put("login", username);
        map.put("password", password);
        map.put("theme", "");

        return Var.valueOf(map);
      }
    }
    else if(userResult.getType() == Var.Type.LIST && userResult.size() > 0) {
      return userResult.get(0);
    }
    else if(userResult.getType() == Var.Type.LIST && userResult.size() == 0) {
      return null;
    }
    else {
      if(userResult.getType() != Var.Type.NULL)
        return userResult;
    }
    return null;
  }

  public User getUser() throws Exception {
    Var resultObject = this.getUserVar();
    if(resultObject != null) {
      Var name = resultObject.getField("name");
      Var username = resultObject.getField("login");
      if (name == null || name.getObject() == null || name.getObjectAsString().isEmpty())
        name = username;
      Var password = resultObject.getField("password");
      Var theme = resultObject.getField("theme");
      Var picture = resultObject.getField("picture");
      return new User(name, username, password, theme, picture);
    }
    return null;
  }

  public Collection<String> getRoles() {
    LinkedHashSet<String> listRoles = new LinkedHashSet<>();
    Var roleResult;
    try {
      if (EventsManager.hasEvent("onGetRoles")) {
        roleResult = EventsManager.executeEventOnTransaction("onGetRoles", Var.valueOf(username));
        if (roleResult.getType() != Var.Type.NULL) {
          if (roleResult.getType() != Var.Type.LIST && !(roleResult.getObject() instanceof DataSource)) {
            listRoles.add(roleResult.getObjectAsString());
          } else {
            for (Object role : roleResult.getObjectAsList()) {
              listRoles.add(Var.valueOf(role).getObjectAsString());
            }
          }
        }
      } else {
        DatabaseQueryManager roleManager = new DatabaseQueryManager("roles");
        roleResult = roleManager.get(username);

        if (roleManager.isDatabase()) {
          if (roleResult.size() > 0) {
            Var roleObject = roleResult.get(0);
            if (roleObject.isNative()) {
              for (Object userRole : roleResult.getObjectAsList()) {
                String roleString = Var.valueOf(userRole).getObjectAsString();
                listRoles.add(roleString);
              }
            } else {
              List userRoles = roleObject.getObjectAsList();
              for (Object userRole : userRoles) {
                Var roleVar = Var.valueOf(userRole).getField("role");
                Var roleName = roleVar.getField("name");
                String roleString = roleName.getObjectAsString();
                listRoles.add(roleString);
              }
            }
          }
        } else {
          if (roleResult.getType() != Var.Type.NULL) {
            if (roleResult.getType() != Var.Type.LIST) {
              listRoles.add(roleResult.getObjectAsString());
            } else {
              for (Object role : roleResult.getObjectAsList()) {
                listRoles.add(Var.valueOf(role).getObjectAsString());
              }
            }
          }
        }
      }
    }
    catch(Exception e) {
      log.error(e.getMessage());
    }
    return listRoles;
  }

  public Set<GrantedAuthority> getAuthorities() {
    return this.getRoles().stream().map(SimpleGrantedAuthority::new).collect(Collectors.toSet());
  }

  public void updateTheme(String theme) throws Exception {
    Var userVar = this.getUserVar();
    if(userVar != null) {
      userVar.setField("theme", theme);
      try {
        DatabaseQueryManager authManager = new DatabaseQueryManager("auth");
        authManager.update(userVar);
      }
      catch(Exception e) {
        log.error(e.getMessage());
      }
    }
  }

  public void updatePassword(String password) throws Exception {
    Var userVar = this.getUserVar();

    if(userVar != null) {
      userVar.setField("password", password);
      try {
        DatabaseQueryManager authManager = new DatabaseQueryManager("auth");
        authManager.update(userVar);
      }
      catch(Exception e) {
        log.error(e.getMessage());
      }
    }
  }
  
  public static void updateDevice(Device device) throws Exception {
    try {
      log.info("Device logged in:  " + device.getId());
        Var deviceResult;
        DatabaseQueryManager deviceManager = new DatabaseQueryManager("device");
        deviceResult = deviceManager.get(device.getId());
        if (deviceManager.isDatabase()) {
            if (deviceResult.size() > 0) {
                Var deviceObject = deviceResult.get(0);
                deviceObject.setField("token", device.getToken());
                deviceObject.setField("appVersion", device.getAppVersion());  
                deviceObject.setField("platformVersion", device.getPlatformVersion());                        
                deviceManager.update(deviceObject);
            } else {
                Map<String, String> map = new LinkedHashMap<>();
                map.put("id", device.getId());
                map.put("platform", device.getPlatform());
                map.put("platformVersion", device.getPlatformVersion());
                map.put("appName", device.getAppName());
                map.put("appVersion", device.getAppVersion());
                map.put("token", device.getToken());
                map.put("model", device.getModel());
                deviceManager.insert(Var.valueOf(map));
            }
        }
    } catch (Exception e) {
        log.error(e.getMessage(), e);
    }
}

}