package cronapp.framework.authentication.social;

import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.Properties;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.env.Environment;
import org.springframework.social.UserIdSource;
import org.springframework.social.config.annotation.*;
import org.springframework.social.connect.ConnectionFactoryLocator;
import org.springframework.social.connect.UsersConnectionRepository;
import org.springframework.social.connect.mem.InMemoryUsersConnectionRepository;
import org.springframework.social.connect.web.ProviderSignInController;
import org.springframework.social.connect.web.SessionUserIdSource;
import org.springframework.social.facebook.connect.FacebookConnectionFactory;
import org.springframework.social.github.connect.GitHubConnectionFactory;
import org.springframework.social.google.connect.GoogleConnectionFactory;
import org.springframework.social.linkedin.connect.LinkedInConnectionFactory;

import cronapi.util.Operations;
import cronapp.framework.api.EventsManager;

@Configuration
@EnableSocial
public class SocialConfig implements SocialConfigurer {
  
  @Autowired(required = false)
  private SocialSignInAdapter socialSignInAdapter;
  
  @Autowired(required = false)
  private SocialConnectionSignup socialConnectionSignup;

  private static Properties PROPERTIES;

  static {
    PROPERTIES = loadProperties();
  }
  
  private static Properties loadProperties() {
    ClassLoader classLoader = EventsManager.class.getClassLoader();
    try (InputStream stream = classLoader.getResourceAsStream("social.properties")) {
      try (InputStreamReader reader = new InputStreamReader(stream)) {
        Properties properties = new Properties();
        properties.load(reader);
        return properties;
      }
    }
    catch(Exception e) {
      // Abafa
    }
    return new Properties();
  }
  
  public static Properties getProperties() {
    if(Operations.IS_DEBUG) {
      return loadProperties();
    }
    else {
      return PROPERTIES;
    }
  }
  
  public static boolean isEnabled(String key) {
    return "true".equals(getProperties().getProperty(key));
  }

  public static boolean isAutoSignUp() {
    return "true".equals(getProperties().getProperty("autoSignUp"));
  }
  
  @Override
  public void addConnectionFactories(ConnectionFactoryConfigurer connectionFactoryConfigurer, Environment environment) {
    if(isEnabled("facebook")) {
      try {
        connectionFactoryConfigurer.addConnectionFactory(new FacebookConnectionFactory(
                getProperties().getProperty("facebook.appId"), getProperties().getProperty("facebook.appSecret")));
      }
      catch(Exception e) {
        // Abafa
      }
    }
    if(isEnabled("github")) {
      try {
        connectionFactoryConfigurer.addConnectionFactory(new GitHubConnectionFactory(
                getProperties().getProperty("github.appId"), getProperties().getProperty("github.appSecret")));
      }
      catch(Exception e) {
        // Abafa
      }
    }
    if(isEnabled("linkedin")) {
      try {
        connectionFactoryConfigurer.addConnectionFactory(new LinkedInConnectionFactory(
                getProperties().getProperty("linkedin.appId"), getProperties().getProperty("linkedin.appSecret")));
      }
      catch(Exception e) {
        // Abafa
      }
    }
    if(isEnabled("google")) {
      try {
        connectionFactoryConfigurer.addConnectionFactory(new GoogleConnectionFactory(
                getProperties().getProperty("google.appId"), getProperties().getProperty("google.appSecret")));
      }
      catch(Exception e) {
        // Abafa
      }
    }
  }

  @Override
  public UserIdSource getUserIdSource() {
    return new SessionUserIdSource();
  }

  @Bean
  public ProviderSignInController providerSignInController(ConnectionFactoryLocator connectionFactoryLocator,
                                                           UsersConnectionRepository usersConnectionRepository) {
    ((InMemoryUsersConnectionRepository)usersConnectionRepository).setConnectionSignUp(socialConnectionSignup);

    return new ProviderSignInController(connectionFactoryLocator, usersConnectionRepository, socialSignInAdapter);
  }
  
  @Override
  public UsersConnectionRepository getUsersConnectionRepository(ConnectionFactoryLocator connectionFactoryLocator) {
    return new InMemoryUsersConnectionRepository(connectionFactoryLocator);
  }
  
}
