package cronapp.framework.authentication.token;

import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import org.springframework.mobile.device.Device;
import org.springframework.security.core.userdetails.UserDetails;

import cronapp.framework.api.ApiManager;
import cronapp.framework.api.User;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SignatureAlgorithm;

class TokenUtils {
  
  private static final String secret = "9SyECk96oDsTmXfogIieDI0cD/8FpnojlYSUJT5U9I/FGVmBz5oskmjOR8cbXTvoPjX+Pq/T/b1PqpHX0lYm0oCBjXWICA==";
  
  private static final String AUDIENCE_MOBILE = "mobile";
  
  private static final String AUDIENCE_TABLET = "tablet";
  
  static final String AUTH_HEADER_NAME = "X-AUTH-TOKEN";
  
  static String getUsernameFromToken(String token) {
    String username;
    try {
      Claims claims = getClaimsFromToken(token);
      username = claims.getSubject();
    }
    catch(Exception e) {
      username = null;
    }
    return username;
  }
  
  private static Date getCreatedDateFromToken(String token) {
    Date created;
    try {
      Claims claims = getClaimsFromToken(token);
      created = new Date((Long)claims.get("created"));
    }
    catch(Exception e) {
      created = null;
    }
    return created;
  }
  
  static Date getExpirationDateFromToken(String token) {
    Date expiration;
    try {
      Claims claims = getClaimsFromToken(token);
      expiration = claims.getExpiration();
    }
    catch(Exception e) {
      expiration = null;
    }
    return expiration;
  }
  
  private static String getAudienceFromToken(String token) {
    String audience;
    try {
      Claims claims = getClaimsFromToken(token);
      audience = (String)claims.get("audience");
    }
    catch(Exception e) {
      audience = null;
    }
    return audience;
  }
  
  private static Claims getClaimsFromToken(String token) {
    Claims claims;
    try {
      claims = Jwts.parser().setSigningKey(secret).parseClaimsJws(token).getBody();
    }
    catch(Exception e) {
      claims = null;
    }
    return claims;
  }
  
  private static Date generateCurrentDate() {
    return new Date(System.currentTimeMillis());
  }
  
  private static Date generateExpirationDate() {
    Long expiration = 3600L;
    return new Date(System.currentTimeMillis() + expiration * 1000);
  }
  
  private static boolean isTokenExpired(String token) {
    Date expiration = getExpirationDateFromToken(token);
    return expiration.before(generateCurrentDate());
  }
  
  private static boolean isCreatedBeforeLastPasswordReset(Date created, Date lastPasswordReset) {
    return (lastPasswordReset != null && created.before(lastPasswordReset));
  }
  
  private static String generateAudience(Device device) {
    String audience = "unknown";
    if(device.isNormal()) {
      audience = "web";
    }
    else if(device.isTablet()) {
      audience = AUDIENCE_TABLET;
    }
    else if(device.isMobile()) {
      audience = AUDIENCE_MOBILE;
    }
    return audience;
  }
  
  private static boolean ignoreTokenExpiration(String token) {
    String audience = getAudienceFromToken(token);
    return (AUDIENCE_TABLET.equals(audience) || AUDIENCE_MOBILE.equals(audience));
  }
  
  static String generateToken(UserDetails userDetails, Device device) {
    Map<String, Object> claims = new HashMap<>();
    claims.put("sub", userDetails.getUsername());
    claims.put("audience", generateAudience(device));
    claims.put("created", generateCurrentDate());
    return generateToken(claims);
  }
  
  private static String generateToken(Map<String, Object> claims) {
    return Jwts.builder().setClaims(claims).setExpiration(generateExpirationDate())
            .signWith(SignatureAlgorithm.HS512, secret).compact();
  }
  
  static boolean canTokenBeRefreshed(String token, Date lastPasswordReset) {
    Date created = getCreatedDateFromToken(token);
    return ((isCreatedBeforeLastPasswordReset(created, lastPasswordReset)) &&
            (!(isTokenExpired(token)) || ignoreTokenExpiration(token)));
  }
  
  static String refreshToken(String token) {
    String refreshedToken;
    try {
      Claims claims = getClaimsFromToken(token);
      claims.put("created", generateCurrentDate());
      refreshedToken = generateToken(claims);
    }
    catch(Exception e) {
      refreshedToken = null;
    }
    return refreshedToken;
  }
  
  static boolean validateToken(String name, String token) {
    ApiManager apiManager = ApiManager.byUser(name);
    User user = apiManager.getUser();
    if(user == null) {
      return false;
    }
    else {
      String usernameFromToken = getUsernameFromToken(token);
      return (usernameFromToken.equals(user.getUsername()) && !(isTokenExpired(token)));
    }
  }
  
}
