package cronapp.framework.authentication.normal;

import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.Collection;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.authentication.AuthenticationServiceException;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.userdetails.User;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.stereotype.Component;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;

import cronapp.framework.api.ApiManager;
import cronapp.framework.api.RecaptchaValidator;
import cronapp.framework.authentication.social.SocialConfig;
import cronapp.framework.i18n.Messages;
import cronapp.framework.tenant.TenantComponent;

@Component
public class AuthenticationConfigurer implements AuthenticationProvider {
  
  private static final Logger log = LoggerFactory.getLogger(AuthenticationConfigurer.class);
  
  @Autowired
  private HttpServletRequest request;

  @Autowired(required = false)
  private TenantComponent tenantComponent;
  
  @Override
  @SuppressWarnings("unchecked")
  public Authentication authenticate(Authentication authentication) throws AuthenticationException {
    try {
      Messages.set(request.getLocale());
      String name = authentication.getName();
      boolean isOauth = authentication.getAuthorities().contains(new SimpleGrantedAuthority("#OAUTH#"));
      String rawPassword = authentication.getCredentials().toString();
      String type = null;
      boolean autoSignUp = false;

      if (isOauth) {
        //Enviando o provider pelo password
        type = rawPassword;
        autoSignUp = SocialConfig.isAutoSignUp();
      }

      if (name.equals("#OAUTH#")) {
        name = (String) request.getSession().getAttribute("#OAUTH#USER");
        isOauth = true;
        request.getSession().removeAttribute("#OAUTH#USER");
      }
	
      ApiManager apiManager = ApiManager.byUserAndPassword(name, rawPassword, type, autoSignUp, authentication.getDetails() instanceof JsonObject ? (JsonObject) authentication.getDetails() : null);
      
      if (authentication instanceof UsernamePasswordAuthenticationToken) {
        ((UsernamePasswordAuthenticationToken) authentication).setDetails(null);
      }
      
      try {
        cronapp.framework.api.User user = apiManager.getUser();
        if (user == null) {
          throw new UsernameNotFoundException(Messages.getString("UserNotFound"));
        } else {
          String password = user.getPassword();
          if (isOauth || apiManager.passwordMatches(rawPassword, password)) {
            Collection<String> listRoles = apiManager.getRoles();
            Set<GrantedAuthority> authorities = apiManager.getAuthorities();
            User userDetails = new User(name, rawPassword, false, false, false, false, authorities);
            UsernamePasswordAuthenticationToken userToken = new UsernamePasswordAuthenticationToken(userDetails, password, authorities);
            userToken.setDetails(userDetails);

            String theme = user.getTheme();
            if (theme == null)
              theme = "";

            HttpSession session = request.getSession();
            session.setAttribute("theme", theme);

            request.setAttribute("userDetails", user);

            if (tenantComponent != null)
              tenantComponent.authenticationTenant(user.getUsername());

            return userToken;
          } else {
            throw new BadCredentialsException(Messages.getString("UserOrPassordInvalids"));
          }
        }
      } catch (Exception e) {
        log.error(Messages.getString("AuthError", e.getMessage()), e);
        throw new AuthenticationServiceException(Messages.getString("AuthError", e.getMessage()));
      }
    } finally {
      Messages.remove();
    }
  }
  
  @Override
  public boolean supports(Class<?> authentication) {
    return authentication.equals(UsernamePasswordAuthenticationToken.class);
  }
  
}
