package cronapp.framework.api;

import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.Set;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;

import cronapi.Var;
import cronapi.database.DatabaseQueryManager;
import cronapp.framework.authentication.normal.AuthenticationConfigurer;

public final class ApiManager {
  
  private static final Logger log = LoggerFactory.getLogger(AuthenticationConfigurer.class);
  
  private final String username;
  
  private final DatabaseQueryManager authManager;
  
  private ApiManager(String username) {
    this.username = username;
    this.authManager = new DatabaseQueryManager("auth");
  }
  
  public static ApiManager byUser(String username) {
    return new ApiManager(username);
  }
  
  private Var getUserVar() {
    Var userResult;
    try {
      userResult = authManager.get(username);
      if(userResult.size() > 0) {
        return userResult.get(0);
      }
    }
    catch(Exception e) {
      log.error(e.getMessage());
    }
    return null;
  }
  
  public User getUser() {
    try {
      Var resultObject = this.getUserVar();
      if(resultObject != null) {
        Var username = resultObject.getField("login");
        Var password = resultObject.getField("password");
        Var theme = resultObject.getField("theme");
        return new User(username, password, theme);
      }
    }
    catch(Exception e) {
      log.error(e.getMessage());
    }
    return null;
  }
  
  public Collection<String> getRoles() {
    LinkedHashSet<String> listRoles = new LinkedHashSet<>();
    DatabaseQueryManager roleManager = new DatabaseQueryManager("roles");
    Var roleResult;
    try {
      roleResult = roleManager.get(username);
      if(roleResult.size() > 0) {
        Var roleObject = roleResult.get(0);
        LinkedList<Var> userRoles = roleObject.getObjectAsList();
        for(Var userRole : userRoles) {
          Var roleVar = userRole.getField("role");
          Var roleName = roleVar.getField("name");
          String roleString = roleName.getObjectAsString();
          listRoles.add(roleString);
        }
      }
    }
    catch(Exception e) {
      log.error(e.getMessage());
    }
    return listRoles;
  }
  
  public Set<GrantedAuthority> getAuthorities() {
    return this.getRoles().stream().map(SimpleGrantedAuthority::new).collect(Collectors.toSet());
  }
  
  public void updateTheme(String theme) {
    Var userVar = this.getUserVar();
    if(userVar != null) {
      userVar.setField("theme", theme);
      try {
        this.authManager.update(userVar);
      }
      catch(Exception e) {
        log.error(e.getMessage());
      }
    }
  }
  
  public void updatePassword(String password) {
    Var userVar = this.getUserVar();
    if(userVar != null) {
      userVar.setField("password", password);
      try {
        this.authManager.update(userVar);
      }
      catch(Exception e) {
        log.error(e.getMessage());
      }
    }
  }
  
}
