package cronapp.framework.authentication.token;

import java.util.Date;
import java.util.Set;
import java.util.StringJoiner;

import javax.servlet.http.HttpServletRequest;

import cronapp.framework.authentication.social.SocialConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.mobile.device.Device;
import org.springframework.security.authentication.AuthenticationServiceException;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UserDetailsService;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.web.bind.annotation.*;

import cronapi.Var;
import cronapp.framework.api.ApiManager;
import cronapp.framework.api.EventsManager;
import cronapp.framework.api.User;
import cronapp.framework.i18n.Messages;
import cronapp.framework.tenant.TenantComponent;

@RestController
@RequestMapping("auth")
public class AuthenticationController {
  
  private static final Logger log = LoggerFactory.getLogger(AuthenticationController.class);
  
  private final PasswordEncoder passwordEncoder = new BCryptPasswordEncoder();
  
  @Autowired
  private UserDetailsService userDetailsService;
  
  @Autowired(required = false)
  private TenantComponent tenantComponent;

  @RequestMapping(method = RequestMethod.POST)
  public ResponseEntity<AuthenticationResponse> authenticationRequest(@RequestParam String username, String password, Device device, @RequestHeader(name = "X-AUTH-TOKEN", required = false) String token) {
    return auth(username, password, device, "local", token);
  }

  public ResponseEntity<AuthenticationResponse> auth(String username, String password, Device device, String provider, String authToken)
          throws AuthenticationException {


    if (authToken != null) {
      String localProvider = TokenUtils.getProviderFromToken(authToken);
      if (provider != null && !"local".equals(localProvider) && username.equals("#OAUTH#")) {
        username = TokenUtils.getUsernameFromToken(authToken);
        provider = localProvider;
      }
    }

    boolean isOauth = !"local".equals(provider);
    ApiManager apiManager = ApiManager.byUserAndPassword(username, password, provider, SocialConfig.isAutoSignUp());

    try {
      User user = apiManager.getUser();
      if (user == null) {
        throw new UsernameNotFoundException(Messages.getString("UserNotFound"));
      } else {
        if (isOauth || apiManager.passwordMatches(password, user.getPassword())) {

          Set<GrantedAuthority> authorities = apiManager.getAuthorities();
          UserDetails userDetails = new org.springframework.security.core.userdetails.User(username, "password", true, true, true, true, authorities);

          String token = TokenUtils.generateToken(userDetails, device, provider);
          Date expires = TokenUtils.getExpirationDateFromToken(token);

          SecurityContextHolder.getContext()
              .setAuthentication(new UsernamePasswordAuthenticationToken(userDetails, "password", authorities));

          StringJoiner roles = new StringJoiner(",");

          roles.add("Public");
          roles.add("Authenticated");

          boolean root = false;
          for (GrantedAuthority role: authorities) {
            roles.add(role.getAuthority());
            if (role.getAuthority().equalsIgnoreCase("Administrators")) {
              root = true;
            }
          }

          if (tenantComponent != null)
            tenantComponent.authenticationTenant(user.getUsername());

          User userWithoutPass = user.resetPassword();

          if(EventsManager.hasEvent("onLogin")) {
            EventsManager.executeEventOnTransaction("onLogin", Var.valueOf(username));
          }

          return ResponseEntity.ok(new AuthenticationResponse(userWithoutPass, token, expires.getTime(), roles.toString(), root));
        } else {
          throw new BadCredentialsException(Messages.getString("UserOrPassordInvalids"));
        }
      }
    } catch (Exception e) {
      log.error(Messages.getString("AuthError", e.getMessage()), e);
      throw new AuthenticationServiceException(Messages.getString("AuthError", e.getMessage()));
    }
  }

  @RequestMapping(value = "refresh", method = RequestMethod.GET)
  public ResponseEntity<?> authenticationRequest(HttpServletRequest request) {
    String tokenHeader = TokenUtils.AUTH_HEADER_NAME;
    String token = request.getHeader(tokenHeader);
    Date expires = TokenUtils.getExpirationDateFromToken(token);
    if(TokenUtils.canTokenBeRefreshed(token, expires)) {
      String refreshedToken = TokenUtils.refreshToken(token);
      expires = TokenUtils.getExpirationDateFromToken(token);
      String username = TokenUtils.getUsernameFromToken(token);
      
      ApiManager apiManager = ApiManager.byUser(username);

      StringJoiner roles = new StringJoiner(",");

      roles.add("Public");
      roles.add("Authenticated");

      boolean root = false;
      for (GrantedAuthority role: apiManager.getAuthorities()) {
        roles.add(role.getAuthority());
        if (role.getAuthority().equalsIgnoreCase("Administrators")) {
          root = true;
        }
      }

      User userWithoutPass = new User(username);

      AuthenticationResponse authenticationResponse = new AuthenticationResponse(userWithoutPass, refreshedToken,
              expires.getTime(), roles.toString(), root);

      return ResponseEntity.ok(authenticationResponse);
    }
    else {
      return ResponseEntity.badRequest().body(null);
    }
  }
  
}
