package cronapp.framework.authentication.token;

import cronapi.Var;
import cronapp.framework.api.EventsManager;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.security.config.annotation.authentication.builders.AuthenticationManagerBuilder;
import org.springframework.security.config.annotation.method.configuration.EnableGlobalMethodSecurity;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter;
import org.springframework.security.config.http.SessionCreationPolicy;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.userdetails.UserDetailsService;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.security.web.authentication.UsernamePasswordAuthenticationFilter;

import cronapp.framework.authentication.security.Permission;
import org.springframework.security.web.authentication.logout.LogoutSuccessHandler;

import javax.servlet.http.HttpServletResponse;

@Configuration
@EnableWebSecurity
@EnableGlobalMethodSecurity(prePostEnabled = true)
public class WebSecurityConfiguration extends WebSecurityConfigurerAdapter {
  
  @Autowired
  private Permission permission;
  
  @Autowired
  private EntryPointUnauthorizedHandler unauthorizedHandler;
  
  @Autowired
  private UserDetailsService userDetailsService;
  
  @Autowired
  public void configureAuthentication(AuthenticationManagerBuilder authenticationManagerBuilder) throws Exception {
    authenticationManagerBuilder.userDetailsService(userDetailsService).passwordEncoder(passwordEncoder());
  }
  
  @Bean
  public PasswordEncoder passwordEncoder() {
    return new BCryptPasswordEncoder();
  }
  
  @Bean
  public AuthenticationTokenFilter authenticationTokenFilterBean() throws Exception {
    AuthenticationTokenFilter authenticationTokenFilter = new AuthenticationTokenFilter();
    authenticationTokenFilter.setAuthenticationManager(super.authenticationManagerBean());
    return authenticationTokenFilter;
  }
  
  @Override
  protected void configure(AuthenticationManagerBuilder auth) throws Exception {
    auth.userDetailsService(userDetailsService).passwordEncoder(passwordEncoder());
  }
  
  private LogoutSuccessHandler logoutHandler() {
    return (request, response, authentication) -> {
      String authToken = request.getHeader(TokenUtils.AUTH_HEADER_NAME);
      String username = TokenUtils.getUsernameFromToken(authToken);
      
      if(username != null && SecurityContextHolder.getContext().getAuthentication() == null) {
        if(!TokenUtils.isTokenExpired(authToken)) {
          if(EventsManager.hasEvent("onLogout")) {
            EventsManager.executeEventOnTransaction("onLogout", Var.valueOf(username));
          }
        }
      }

      if(request.getHeader("Accept") == null || !request.getHeader("Accept").contains("json")) {
        response.setStatus(HttpServletResponse.SC_OK);
        response.sendRedirect("/index.html");
      }
      
    };
  }
  
  @Override
  protected void configure(HttpSecurity httpSecurity) throws Exception {
    // configuration
    httpSecurity.csrf().disable().exceptionHandling().authenticationEntryPoint(unauthorizedHandler).and().logout()
            .logoutSuccessHandler(logoutHandler()).and().sessionManagement()
            .sessionCreationPolicy(SessionCreationPolicy.STATELESS);
    
    // security permission
    permission.loadSecurityPermission(httpSecurity);
    
    // x-frame-options disable
    httpSecurity.headers().frameOptions().disable().httpStrictTransportSecurity().disable();
    
    // Custom JWT based authentication
    httpSecurity.addFilterBefore(authenticationTokenFilterBean(), UsernamePasswordAuthenticationFilter.class);
  }
  
}
