package cronapp.framework.authentication.token;

import java.util.Collection;
import java.util.Date;

import javax.servlet.http.HttpServletRequest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.mobile.device.Device;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UserDetailsService;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import cronapp.framework.api.ApiManager;
import cronapp.framework.api.User;
import cronapp.framework.authentication.security.SecurityPermission;

@RestController
@RequestMapping("auth")
public class AuthenticationController {
  
  private static final Logger log = LoggerFactory.getLogger(AuthenticationController.class);
  
  private final PasswordEncoder passwordEncoder = new BCryptPasswordEncoder();
    
  private final UserDetailsService userDetailsService;
  
  @Autowired
  public AuthenticationController(UserDetailsService userDetailsService) {
    this.userDetailsService = userDetailsService;
  }
  
  @RequestMapping(method = RequestMethod.POST)
  public ResponseEntity<?> authenticationRequest(@RequestParam String username, String password, Device device)
          throws AuthenticationException {
    
    ApiManager apiManager = ApiManager.byUser(username);
    User user = apiManager.getUser();
    if(user == null) {
      log.error("Usuário não encontrado!");
      throw new UsernameNotFoundException("Usuário não encontrado!");
    }
    else {
      if(passwordEncoder.matches(password, user.getPassword())) {
        UserDetails userDetails = userDetailsService.loadUserByUsername(username);
        String token = TokenUtils.generateToken(userDetails, device);
        Date expires = TokenUtils.getExpirationDateFromToken(token);
        
        Collection<? extends GrantedAuthority> authorities = userDetails.getAuthorities();
        SecurityContextHolder.getContext()
                .setAuthentication(new UsernamePasswordAuthenticationToken(userDetails, "password", authorities));
        
        String roles = authorities.toString().replaceFirst("\\[", "").replaceFirst("\\]", "");
        
        boolean root = roles.contains(SecurityPermission.ROLE_ADMIN_NAME);
        User userWithoutPass = user.resetPassword();
        return ResponseEntity.ok(new AuthenticationResponse(userWithoutPass, token, expires.getTime(), roles, root));
      }
      else {
        throw new BadCredentialsException("Invalid Password!");
      }
    }
  }
  
  @RequestMapping(value = "refresh", method = RequestMethod.GET)
  public ResponseEntity<?> authenticationRequest(HttpServletRequest request) {
    String tokenHeader = TokenUtils.AUTH_HEADER_NAME;
    String token = request.getHeader(tokenHeader);
    Date expires = TokenUtils.getExpirationDateFromToken(token);
    if(TokenUtils.canTokenBeRefreshed(token, expires)) {
      String refreshedToken = TokenUtils.refreshToken(token);
      expires = TokenUtils.getExpirationDateFromToken(token);
      String username = TokenUtils.getUsernameFromToken(token);
      
      ApiManager apiManager = ApiManager.byUser(username);
      User user = apiManager.getUser();
      if(user == null) {
        log.error("Usuário não encontrado!");
        throw new UsernameNotFoundException("Usuário não encontrado!");
      }
      else {
        UserDetails userDetails = userDetailsService.loadUserByUsername(username);
        String roles = userDetails.getAuthorities().toString().replaceFirst("\\[", "").replaceFirst("\\]", "");
        boolean root = roles.contains(SecurityPermission.ROLE_ADMIN_NAME);
        User userWithoutPass = user.resetPassword();
        
        AuthenticationResponse authenticationResponse = new AuthenticationResponse(userWithoutPass, refreshedToken,
                expires.getTime(), roles, root);
        
        return ResponseEntity.ok(authenticationResponse);
      }
    }
    else {
      return ResponseEntity.badRequest().body(null);
    }
  }
  
}
