package cronapp.framework.authentication.normal;

import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import cronapp.framework.i18n.Messages;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.HttpStatus;
import org.springframework.security.authentication.AuthenticationServiceException;
import org.springframework.security.config.annotation.authentication.builders.AuthenticationManagerBuilder;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter;
import org.springframework.security.config.http.SessionCreationPolicy;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.userdetails.User;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.security.web.authentication.AuthenticationFailureHandler;
import org.springframework.security.web.authentication.AuthenticationSuccessHandler;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;

import com.google.gson.Gson;
import com.google.gson.JsonObject;

import cronapp.framework.api.ApiManager;
import cronapp.framework.authentication.security.Permission;

import java.util.StringJoiner;

/**
 * Classe que configura o WebSecurity, possibilitando requerer que o
 * usuário esteja logado para acessar qualquer usuário
 *
 * @author Techne
 */
@Configuration
@EnableWebSecurity
public class AuthorizationConfigurer extends WebSecurityConfigurerAdapter {

  private static final Logger log = LoggerFactory.getLogger(AuthorizationConfigurer.class);

  @Autowired
  private AuthenticationConfigurer authenticationProvider;

  @Autowired
  private Permission permission;

  /**
   * Método que configura o SpringSecurite com o authenticationProvider,
   * responsável também pela criação da base inicial, caso não exista nenhuma
   * permissão cadastrada.
   */
  @Override
  protected void configure(AuthenticationManagerBuilder auth) throws Exception {
    auth.authenticationProvider(authenticationProvider);
  }

  /**
   * Configurações default para WebSecurity
   */
  @Override
  protected void configure(HttpSecurity http) throws Exception {

    // post sem csrf
    http.csrf().disable();

    // session manager
    http.sessionManagement()
            .maximumSessions(1)
            .maxSessionsPreventsLogin(false)
            .expiredUrl("/index.html")
            .and()
            .sessionCreationPolicy(SessionCreationPolicy.IF_REQUIRED)
            .invalidSessionUrl("/index.html");

    // load Security Permission
    permission.loadSecurityPermission(http);

    // login/logout
    http.formLogin()
            .loginProcessingUrl("/auth")
            .loginPage("/index.html")
            .successHandler(successHandler())
            .failureHandler(failureHandler())
            .and()
            .logout()
            .logoutRequestMatcher(new AntPathRequestMatcher("/logout"))
            .invalidateHttpSession(true);

    // x-frame-options disable
    http.headers()
            .frameOptions()
            .disable()
            .httpStrictTransportSecurity()
            .disable();
  }

  /**
   * Handler para sucesso de autorização
   */
  @SuppressWarnings("unchecked")
  private AuthenticationSuccessHandler successHandler() {
    return (request, response, authentication) -> {

      HttpSession session = request.getSession();
      
      User authUser = (User)SecurityContextHolder.getContext().getAuthentication().getPrincipal();
      
      String username = authUser.getUsername();
      session.setAttribute("username", username);
      session.setAttribute("authorities", authentication.getAuthorities());

      response.setStatus(HttpServletResponse.SC_OK);

      StringJoiner roles = new StringJoiner(",");

      roles.add("Public");
      roles.add("Authenticated");

      boolean root = false;
      for (GrantedAuthority role: authUser.getAuthorities()) {
        roles.add(role.getAuthority());
        if (role.getAuthority().equalsIgnoreCase("Administrators")) {
          root = true;
        }
      }

      try {
        cronapp.framework.api.User user = (cronapp.framework.api.User) request.getAttribute("userDetails");

        Gson gson = new Gson();
        JsonObject json = new JsonObject();
        json.add("user", gson.toJsonTree(user.resetPassword()));
        json.addProperty("roles", roles.toString());
        json.addProperty("theme", ""+session.getAttribute("theme"));
        json.addProperty("root", root);

        response.getOutputStream().print(json.toString());
        response.setHeader("Content-Type", "application/json");
      }
      catch(Exception e) {
        log.error(Messages.getString("AuthError", e.getMessage()), e);
        throw new AuthenticationServiceException(Messages.getString("AuthError", e.getMessage()));
      }
    };
  }
  
  /**
   * Handler para falha de autorização
   */
  private AuthenticationFailureHandler failureHandler() {
    return (request, response, failureHandler) -> response.setStatus(HttpStatus.UNAUTHORIZED.value());
  }
  
}
