package cronapp.framework.authentication.normal;

import java.util.Collection;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.authentication.AuthenticationServiceException;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.userdetails.User;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.stereotype.Component;

import cronapp.framework.api.ApiManager;
import cronapp.framework.i18n.Messages;
import cronapp.framework.tenant.TenantComponent;

@Component
public class AuthenticationConfigurer implements AuthenticationProvider {
  
  private static final Logger log = LoggerFactory.getLogger(AuthenticationConfigurer.class);
  
  @Autowired
  private HttpServletRequest request;

  @Autowired(required = false)
  private TenantComponent tenantComponent;
  
  @Override
  @SuppressWarnings("unchecked")
  public Authentication authenticate(Authentication authentication) throws AuthenticationException {
    try {
      Messages.set(request.getLocale());
      String name = authentication.getName();
      String rawPassword = authentication.getCredentials().toString();
      ApiManager apiManager = ApiManager.byUserAndPassword(name, rawPassword);
      try {
        cronapp.framework.api.User user = apiManager.getUser();
        if (user == null) {
          throw new UsernameNotFoundException(Messages.getString("UserNotFound"));
        } else {
          String password = user.getPassword();
          if (apiManager.passwordMatches(rawPassword, password)) {
            Collection<String> listRoles = apiManager.getRoles();
            Set<GrantedAuthority> authorities = apiManager.getAuthorities();
            User userDetails = new User(name, rawPassword, false, false, false, false, authorities);
            UsernamePasswordAuthenticationToken userToken = new UsernamePasswordAuthenticationToken(userDetails, password, authorities);
            userToken.setDetails(userDetails);

            String theme = user.getTheme();
            if (theme == null)
              theme = "";

            HttpSession session = request.getSession();
            session.setAttribute("theme", theme);

            request.setAttribute("userDetails", user);

            if (tenantComponent != null)
              tenantComponent.authenticationTenant(user.getUsername());

            return userToken;
          } else {
            throw new BadCredentialsException(Messages.getString("UserOrPassordInvalids"));
          }
        }
      } catch (Exception e) {
        log.error(Messages.getString("AuthError", e.getMessage()), e);
        throw new AuthenticationServiceException(Messages.getString("AuthError", e.getMessage()));
      }
    } finally {
      Messages.remove();
    }
  }
  
  @Override
  public boolean supports(Class<?> authentication) {
    return authentication.equals(UsernamePasswordAuthenticationToken.class);
  }
  
}
