package cronapp.framework.api;

import java.util.*;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;

import cronapi.Var;
import cronapi.database.DatabaseQueryManager;
import cronapp.framework.authentication.normal.AuthenticationConfigurer;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;
import org.springframework.security.crypto.password.PasswordEncoder;

public final class ApiManager {

  private static final Logger log = LoggerFactory.getLogger(AuthenticationConfigurer.class);

  private final String username;

  private final String password;

  private final DatabaseQueryManager authManager;

  private final PasswordEncoder passwordEncoder = new BCryptPasswordEncoder();

  private ApiManager(String username, String password) {
    this.username = username;
    this.password = password;
    this.authManager = new DatabaseQueryManager("auth");
  }

  public boolean passwordMatches(CharSequence rawPassword, String password) {
    if (this.authManager.isDatabase()) {
      return passwordEncoder.matches(rawPassword, password);
    } else {
      return true;
    }
  }

  public static ApiManager byUser(String username) {
    return new ApiManager(username, null);
  }

  public static ApiManager byUserAndPassword(String username, String password) {
    return new ApiManager(username, password);
  }

  private Var getUserVar() throws Exception {
    Var userResult = authManager.get(username, password);

    if (userResult.getType() == Var.Type.BOOLEAN) {
      if (userResult.getObjectAsBoolean()) {
        Map<String, String> map = new LinkedHashMap<>();
        map.put("login", username);
        map.put("password", password);
        map.put("theme", "");

        return Var.valueOf(map);
      }
    }
    else if(userResult.getType() == Var.Type.LIST && userResult.size() > 0) {
      return userResult.get(0);
    } else {
      return userResult;
    }

    return null;
  }

  public User getUser() throws Exception {
    Var resultObject = this.getUserVar();
    if(resultObject != null) {
      Var username = resultObject.getField("login");
      Var password = resultObject.getField("password");
      Var theme = resultObject.getField("theme");
      return new User(username, password, theme);
    }
    return null;
  }

  public Collection<String> getRoles() {
    LinkedHashSet<String> listRoles = new LinkedHashSet<>();
    DatabaseQueryManager roleManager = new DatabaseQueryManager("roles");
    Var roleResult;
    try {
      roleResult = roleManager.get(username);

      if (roleManager.isDatabase()) {
        if(roleResult.size() > 0) {
          Var roleObject = roleResult.get(0);
          LinkedList<Var> userRoles = roleObject.getObjectAsList();
          for(Var userRole : userRoles) {
            Var roleVar = userRole.getField("role");
            Var roleName = roleVar.getField("name");
            String roleString = roleName.getObjectAsString();
            listRoles.add(roleString);
          }
        }
      } else {
        if(roleResult.getType() != Var.Type.NULL) {
          if (roleResult.getType() != Var.Type.LIST) {
            listRoles.add(roleResult.getObjectAsString());
          }
        } else {
          for(Var role : roleResult.getObjectAsList()) {
            listRoles.add(role.getObjectAsString());
          }
        }
      }


    }
    catch(Exception e) {
      log.error(e.getMessage());
    }
    return listRoles;
  }

  public Set<GrantedAuthority> getAuthorities() {
    return this.getRoles().stream().map(SimpleGrantedAuthority::new).collect(Collectors.toSet());
  }

  public void updateTheme(String theme) throws Exception {
    Var userVar = this.getUserVar();
    if(userVar != null) {
      userVar.setField("theme", theme);
      try {
        this.authManager.update(userVar);
      }
      catch(Exception e) {
        log.error(e.getMessage());
      }
    }
  }

  public void updatePassword(String password) throws Exception {
    Var userVar = this.getUserVar();
    if(userVar != null) {
      userVar.setField("password", password);
      try {
        this.authManager.update(userVar);
      }
      catch(Exception e) {
        log.error(e.getMessage());
      }
    }
  }

}