package cronapp.framework.authentication.normal;

import java.util.Collection;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.authentication.AuthenticationServiceException;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.userdetails.User;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.stereotype.Component;

import cronapp.framework.api.ApiManager;
import cronapp.framework.tenant.TenantComponent;

@Component
public class AuthenticationConfigurer implements AuthenticationProvider {
  
  private static final Logger log = LoggerFactory.getLogger(AuthenticationConfigurer.class);
  
  private final PasswordEncoder passwordEncoder = new BCryptPasswordEncoder();

  @Autowired
  private HttpServletRequest request;

  @Autowired(required = false)
  private TenantComponent tenantComponent;
  
  @Override
  @SuppressWarnings("unchecked")
  public Authentication authenticate(Authentication authentication) throws AuthenticationException {
    String name = authentication.getName();
    String rawPassword = authentication.getCredentials().toString();
    ApiManager apiManager = ApiManager.byUser(name);
    try {
      cronapp.framework.api.User user = apiManager.getUser();
      if(user == null) {
        log.error("Usuário não encontrado!");
        throw new UsernameNotFoundException("Usuário não encontrado!");
      }
      else {
        String password = user.getPassword();
        if(passwordEncoder.matches(rawPassword, password)) {
          Collection<String> listRoles = apiManager.getRoles();
          if(listRoles.isEmpty()) {
            log.error("O usuário " + name + " não possui autorização de acesso.");
            throw new AuthenticationServiceException("O usuário " + name + " não possui autorização de acesso.");
          }
          else {
            Set<GrantedAuthority> authorities = apiManager.getAuthorities();
            User userDetails = new User(name, password, false, false, false, false, authorities);
            UsernamePasswordAuthenticationToken userToken = new UsernamePasswordAuthenticationToken(userDetails, password, authorities);
            userToken.setDetails(userDetails);
            
            String theme = user.getTheme();
            if(theme == null)
              theme = "";
            
            HttpSession session = request.getSession();
            session.setAttribute("theme", theme);
            
            if(tenantComponent != null)
              tenantComponent.authenticationTenant(user.getUsername());
            
            return userToken;
          }
        }
        else {
          log.error("Usuário ou senha incorretos!");
          throw new BadCredentialsException("Usuário ou senha incorreta!");
        }
      }
    }
    catch(Exception e) {
      log.error(e.getMessage());
      throw new AuthenticationServiceException("Erro durante o processo de autenticação:\n".concat(e.getMessage()));
    }
  }
  
  @Override
  public boolean supports(Class<?> authentication) {
    return authentication.equals(UsernamePasswordAuthenticationToken.class);
  }
  
}
