/**
 * Copyright (C) 2018 Alauda
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.alauda.devops.client;

import io.alauda.kubernetes.api.model.*;
import io.alauda.kubernetes.api.model.apiextensions.CustomResourceDefinition;
import io.alauda.kubernetes.api.model.apiextensions.CustomResourceDefinitionList;
import io.alauda.kubernetes.api.model.apiextensions.DoneableCustomResourceDefinition;
import io.alauda.kubernetes.api.model.extensions.DoneablePodSecurityPolicy;
import io.alauda.kubernetes.api.model.extensions.PodSecurityPolicy;
import io.alauda.kubernetes.api.model.extensions.PodSecurityPolicyList;
import io.alauda.devops.client.dsl.*;
import io.alauda.devops.client.dsl.internal.*;
import io.alauda.kubernetes.client.AppsAPIGroupClient;
import io.alauda.kubernetes.client.AutoscalingAPIGroupClient;
import io.alauda.kubernetes.client.RequestConfig;
import io.alauda.kubernetes.client.dsl.AppsAPIGroupDSL;
import io.alauda.kubernetes.client.dsl.AutoscalingAPIGroupDSL;
import io.alauda.kubernetes.client.dsl.FunctionCallable;
import io.alauda.kubernetes.client.dsl.LogWatch;
import io.alauda.kubernetes.client.dsl.NamespaceListVisitFromServerGetDeleteRecreateWaitApplicable;
import io.alauda.kubernetes.client.WithRequestCallable;
import io.alauda.kubernetes.client.dsl.ParameterNamespaceListVisitFromServerGetDeleteRecreateWaitApplicable;
import io.alauda.kubernetes.client.dsl.internal.CustomResourceOperationsImpl;
import io.alauda.kubernetes.client.dsl.internal.PodSecurityPolicyOperationsImpl;
import io.alauda.kubernetes.client.utils.ImpersonatorInterceptor;
import io.alauda.kubernetes.client.utils.Serialization;
import okhttp3.Authenticator;
import okhttp3.OkHttpClient;

import io.alauda.kubernetes.client.BaseClient;
import io.alauda.kubernetes.client.Config;
import io.alauda.kubernetes.client.DefaultKubernetesClient;
import io.alauda.kubernetes.client.ExtensionsAPIGroupClient;
import io.alauda.kubernetes.client.NamespacedKubernetesClient;
import io.alauda.kubernetes.client.KubernetesClientException;
import io.alauda.kubernetes.client.dsl.KubernetesListMixedOperation;
import io.alauda.kubernetes.client.dsl.MixedOperation;
import io.alauda.kubernetes.client.dsl.NonNamespaceOperation;
import io.alauda.kubernetes.client.dsl.PodResource;
import io.alauda.kubernetes.client.dsl.Resource;
import io.alauda.kubernetes.client.dsl.RollableScalableResource;
import io.alauda.kubernetes.client.dsl.NamespaceVisitFromServerGetWatchDeleteRecreateWaitApplicable;
import io.alauda.kubernetes.client.dsl.internal.ComponentStatusOperationsImpl;
import io.alauda.kubernetes.client.dsl.internal.CustomResourceDefinitionOperationsImpl;
import io.alauda.devops.client.internal.AlaudaOAuthInterceptor;

import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Collection;
import java.util.List;
import java.util.Objects;

public class DefaultAlaudaDevOpsClient extends BaseClient implements NamespacedAlaudaDevOpsClient {

  private URL kubernetesUrl;
  private NamespacedKubernetesClient delegate;

  public DefaultAlaudaDevOpsClient() throws KubernetesClientException {
    this(new AlaudaDevOpsConfigBuilder().build());
  }

  public DefaultAlaudaDevOpsClient(String masterUrl) throws KubernetesClientException {
    this(new AlaudaDevOpsConfigBuilder().withMasterUrl(masterUrl).build());
  }

  public DefaultAlaudaDevOpsClient(final Config config) throws KubernetesClientException {
    this(new AlaudaDevOpsConfig(config));
  }

  public DefaultAlaudaDevOpsClient(final AlaudaDevOpsConfig config) throws KubernetesClientException {
    super(config);
    try {
      this.httpClient = clientWithOpenShiftOAuthInterceptor(this.httpClient);
      this.delegate = new DefaultKubernetesClient(this.httpClient, config);
      this.kubernetesUrl = new URL(config.getKubernetesUrl());
    } catch (MalformedURLException e) {
      throw new KubernetesClientException("Could not create client", e);
    }
  }

  public DefaultAlaudaDevOpsClient(OkHttpClient httpClient, AlaudaDevOpsConfig config) throws KubernetesClientException {
    super(httpClient, config);
    try {
      this.delegate = new DefaultKubernetesClient(clientWithOpenShiftOAuthInterceptor(httpClient), config);
      this.kubernetesUrl = new URL(config.getKubernetesUrl());
    } catch (MalformedURLException e) {
      throw new KubernetesClientException("Could not create client", e);
    }
  }

  public static DefaultAlaudaDevOpsClient fromConfig(String config) {
    return new DefaultAlaudaDevOpsClient(Serialization.unmarshal(config, AlaudaDevOpsConfig.class));
  }

  public static DefaultAlaudaDevOpsClient fromConfig(InputStream is) {
    return new DefaultAlaudaDevOpsClient(Serialization.unmarshal(is, AlaudaDevOpsConfig.class));
  }

  private OkHttpClient clientWithOpenShiftOAuthInterceptor(OkHttpClient httpClient) {
    httpClient = httpClient.newBuilder().authenticator(Authenticator.NONE).build();
    OkHttpClient.Builder builder = httpClient.newBuilder();
    builder.interceptors().clear();
    return builder.addInterceptor(new AlaudaOAuthInterceptor(httpClient, AlaudaDevOpsConfig.wrap(getConfiguration())))
      .addInterceptor(new ImpersonatorInterceptor(getConfiguration()))
      .build();
  }

  @Override
  public URL getKubernetesUrl() {
    return kubernetesUrl;
  }

  @Override
  public MixedOperation<ComponentStatus, ComponentStatusList, DoneableComponentStatus, Resource<ComponentStatus, DoneableComponentStatus>> componentstatuses() {
    return new ComponentStatusOperationsImpl(httpClient, getConfiguration());
  }

  @Override
  public ParameterNamespaceListVisitFromServerGetDeleteRecreateWaitApplicable<HasMetadata, Boolean> load(InputStream is) {
    return delegate.load(is);
  }

  @Override
  public NamespaceVisitFromServerGetWatchDeleteRecreateWaitApplicable<HasMetadata, Boolean> resource(HasMetadata item) {
    return delegate.resource(item);
  }

  @Override
  public NamespaceVisitFromServerGetWatchDeleteRecreateWaitApplicable<HasMetadata, Boolean> resource(String s) {
    return delegate.resource(s);
  }

  @Override
  public NamespaceListVisitFromServerGetDeleteRecreateWaitApplicable<HasMetadata, Boolean> resourceList(KubernetesResourceList is) {
    return delegate.resourceList(is);
  }

  @Override
  public NamespaceListVisitFromServerGetDeleteRecreateWaitApplicable<HasMetadata, Boolean> resourceList(HasMetadata... items) {
    return delegate.resourceList(items);
  }

  @Override
  public NamespaceListVisitFromServerGetDeleteRecreateWaitApplicable<HasMetadata, Boolean> resourceList(Collection<HasMetadata> items) {
    return delegate.resourceList(items);
  }

  @Override
  public ParameterNamespaceListVisitFromServerGetDeleteRecreateWaitApplicable<HasMetadata, Boolean> resourceList(String s) {
    return delegate.resourceList(s);
  }

  @Override
  public MixedOperation<Endpoints, EndpointsList, DoneableEndpoints, Resource<Endpoints, DoneableEndpoints>> endpoints() {
    return delegate.endpoints();
  }

  @Override
  public MixedOperation<Event, EventList, DoneableEvent, Resource<Event, DoneableEvent>> events() {
    return delegate.events();
  }

  @Override
  public NonNamespaceOperation<Namespace, NamespaceList, DoneableNamespace, Resource<Namespace, DoneableNamespace>> namespaces() {
    return delegate.namespaces();
  }

  @Override
  public NonNamespaceOperation<Node, NodeList, DoneableNode, Resource<Node, DoneableNode>> nodes() {
    return delegate.nodes();
  }

  @Override
  public NonNamespaceOperation<PersistentVolume, PersistentVolumeList, DoneablePersistentVolume, Resource<PersistentVolume, DoneablePersistentVolume>> persistentVolumes() {
    return delegate.persistentVolumes();
  }

  @Override
  public MixedOperation<PersistentVolumeClaim, PersistentVolumeClaimList, DoneablePersistentVolumeClaim, Resource<PersistentVolumeClaim, DoneablePersistentVolumeClaim>> persistentVolumeClaims() {
    return delegate.persistentVolumeClaims();
  }

  @Override
  public MixedOperation<Pod, PodList, DoneablePod, PodResource<Pod, DoneablePod>> pods() {
    return delegate.pods();
  }

  @Override
  public MixedOperation<ReplicationController, ReplicationControllerList, DoneableReplicationController, RollableScalableResource<ReplicationController, DoneableReplicationController>> replicationControllers() {
    return delegate.replicationControllers();
  }

  @Override
  public MixedOperation<ResourceQuota, ResourceQuotaList, DoneableResourceQuota, Resource<ResourceQuota, DoneableResourceQuota>> resourceQuotas() {
    return delegate.resourceQuotas();
  }

  @Override
  public MixedOperation<Secret, SecretList, DoneableSecret, Resource<Secret, DoneableSecret>> secrets() {
    return delegate.secrets();
  }

  @Override
  public MixedOperation<Service, ServiceList, DoneableService, Resource<Service, DoneableService>> services() {
    return delegate.services();
  }

  @Override
  public MixedOperation<ServiceAccount, ServiceAccountList, DoneableServiceAccount, Resource<ServiceAccount, DoneableServiceAccount>> serviceAccounts() {
    return delegate.serviceAccounts();
  }

  @Override
  public KubernetesListMixedOperation lists() {
    return delegate.lists();
  }

  @Override
  public MixedOperation<ConfigMap, ConfigMapList, DoneableConfigMap, Resource<ConfigMap, DoneableConfigMap>> configMaps() {
    return delegate.configMaps();
  }

  @Override
  public MixedOperation<Pipeline, PipelineList, DoneablePipeline, PipelineResource<Pipeline, DoneablePipeline, String, LogWatch>> pipelines() {
    return new PipelineOperationsImpl(httpClient, AlaudaDevOpsConfig.wrap(getConfiguration()), getNamespace());
  }

  @Override
  public MixedOperation<PipelineConfig, PipelineConfigList, DoneablePipelineConfig, PipelineConfigResource<PipelineConfig, DoneablePipelineConfig, Void, Pipeline>> pipelineConfigs() {
    return new PipelineConfigOperationsImpl(httpClient, AlaudaDevOpsConfig.wrap(getConfiguration()), getNamespace());
  }

  @Override
  public MixedOperation<PipelineTemplate, PipelineTemplateList, DoneablePipelineTemplate,
    PipelineTemplateResource<PipelineTemplate, DoneablePipelineTemplate, Void, PipelineTemplate>> pipelinetemplates() {
    return new PipelineTemplateOperationsImpl(httpClient, AlaudaDevOpsConfig.wrap(getConfiguration()), getNamespace());
  }

  @Override
  public MixedOperation<PipelineTaskTemplate, PipelineTaskTemplateList, DoneablePipelineTaskTemplate,
    PipelineTaskTemplateResource<PipelineTaskTemplate, DoneablePipelineTaskTemplate, Void, PipelineTaskTemplate>> pipelinetasktemplates() {
    return new PipelineTaskTemplateOperationsImpl(httpClient, AlaudaDevOpsConfig.wrap(getConfiguration()), getNamespace());
  }

  @Override
  public NonNamespaceOperation<ClusterPipelineTemplate, ClusterPipelineTemplateList, DoneableClusterPipelineTemplate,
    ClusterPipelineTemplateResource<ClusterPipelineTemplate, DoneableClusterPipelineTemplate, Void, ClusterPipelineTemplate>> clusterPipelinetemplates() {
    return new ClusterPipelineTemplateOperationsImpl(httpClient, AlaudaDevOpsConfig.wrap(getConfiguration()));
  }

  @Override
  public NonNamespaceOperation<ClusterPipelineTaskTemplate, ClusterPipelineTaskTemplateList, DoneableClusterPipelineTaskTemplate,
    ClusterPipelineTaskTemplateResource<ClusterPipelineTaskTemplate, DoneableClusterPipelineTaskTemplate, Void, ClusterPipelineTaskTemplate>> clusterPipelinetasktemplates() {
    return new ClusterPipelineTaskTemplateOperationsImpl(httpClient, AlaudaDevOpsConfig.wrap(getConfiguration()));
  }

  @Override
  public MixedOperation<LimitRange, LimitRangeList, DoneableLimitRange, Resource<LimitRange, DoneableLimitRange>> limitRanges() {
    return delegate.limitRanges();
  }

  @Override
  public <T extends HasMetadata, L extends KubernetesResourceList, D extends Doneable<T>> MixedOperation<T, L, D, Resource<T, D>> customResources(CustomResourceDefinition crd, Class<T> resourceType, Class<L> listClass, Class<D> doneClass) {
    return new CustomResourceOperationsImpl<T,L,D>(httpClient, getConfiguration(), crd, resourceType, listClass, doneClass);
  }

  @Override
  public <T extends HasMetadata, L extends KubernetesResourceList, D extends Doneable<T>> MixedOperation<T, L, D, Resource<T, D>> customResource(CustomResourceDefinition crd, Class<T> resourceType, Class<L> listClass, Class<D> doneClass) {
    return customResources(crd, resourceType, listClass, doneClass);
  }

  @Override
  public NonNamespaceOperation<CustomResourceDefinition, CustomResourceDefinitionList, DoneableCustomResourceDefinition, Resource<CustomResourceDefinition, DoneableCustomResourceDefinition>> customResourceDefinitions() {
    return new CustomResourceDefinitionOperationsImpl(httpClient, getConfiguration());
  }

  @Override
  public NonNamespaceOperation<Project, ProjectList, DoneableProject, Resource<Project, DoneableProject>> projects() {
    return new ProjectOperationsImpl(httpClient, AlaudaDevOpsConfig.wrap(getConfiguration()));
  }

  @Override
  public NonNamespaceOperation<Jenkins, JenkinsList, DoneableJenkins, Resource<Jenkins, DoneableJenkins>> jenkins() {
    return new JenkinsOperationsImpl(httpClient, AlaudaDevOpsConfig.wrap(getConfiguration()));
  }

  @Override
  public MixedOperation<JenkinsBinding, JenkinsBindingList, DoneableJenkinsBinding, Resource<JenkinsBinding, DoneableJenkinsBinding>> jenkinsBindings() {
    return new JenkinsBindingOperationsImpl(httpClient, AlaudaDevOpsConfig.wrap(getConfiguration()), getNamespace());
  }

  @Override
  public MixedOperation<CodeRepository, CodeRepositoryList, DoneableCodeRepository, Resource<CodeRepository, DoneableCodeRepository>> codeRepositories() {
    return new CodeRepositoryOperationsImpl(httpClient, AlaudaDevOpsConfig.wrap(getConfiguration()), getNamespace());
  }

  @Override
  public MixedOperation<CodeRepoBinding, CodeRepoBindingList, DoneableCodeRepoBinding, Resource<CodeRepoBinding, DoneableCodeRepoBinding>> codeRepoBindings() {
    return new CodeRepoBindingOperationsImpl(httpClient, AlaudaDevOpsConfig.wrap(getConfiguration()), getNamespace());
  }

  @Override
  public MixedOperation<CodeRepoService, CodeRepoServiceList, DoneableCodeRepoService, Resource<CodeRepoService, DoneableCodeRepoService>> codeRepoServices() {
    return new CodeRepoServiceOperationsImpl(httpClient, AlaudaDevOpsConfig.wrap(getConfiguration()), getNamespace());
  }

  @Override
  public MixedOperation<StorageClass, StorageClassList, DoneableStorageClass, Resource<StorageClass, DoneableStorageClass>> storageClasses() {
    return delegate.storageClasses();
  }

  @Override
  public MixedOperation<PodSecurityPolicy, PodSecurityPolicyList, DoneablePodSecurityPolicy, Resource<PodSecurityPolicy, DoneablePodSecurityPolicy>> podSecurityPolicies() {
    return new PodSecurityPolicyOperationsImpl(httpClient, getConfiguration(), getNamespace());
  }

  @Override
  public NamespacedAlaudaDevOpsClient inNamespace(String namespace) {
    AlaudaDevOpsConfig updated = new AlaudaDevOpsConfigBuilder(new AlaudaDevOpsConfig(getConfiguration()))
      .withMasterUrl(kubernetesUrl.toString())
      .withNamespace(namespace)
      .build();
    return new DefaultAlaudaDevOpsClient(httpClient, updated);
  }

  @Override
  public NamespacedAlaudaDevOpsClient inAnyNamespace() {
    return inNamespace(null);
  }

  @Override
  public ExtensionsAPIGroupClient extensions() {
    return adapt(ExtensionsAPIGroupClient.class);
  }

  @Override
  public AppsAPIGroupDSL apps() {
    return adapt(AppsAPIGroupClient.class);
  }

  @Override
  public AutoscalingAPIGroupDSL autoscaling() {
    return adapt(AutoscalingAPIGroupClient.class);
  }

  @Override
  public FunctionCallable<NamespacedAlaudaDevOpsClient> withRequestConfig(RequestConfig requestConfig) {
    return new WithRequestCallable<NamespacedAlaudaDevOpsClient>(this, requestConfig);
  }

  @Override
  public boolean supportAlaudaAPIGroup(String apiGroup) {
    String apiGroupPath = "/apis/" + apiGroup;
    RootPaths rootPaths = rootPaths();
    if (rootPaths != null) {
      List<String> paths = rootPaths.getPaths();
      if (paths != null) {
        for (String path : paths) {
          if (Objects.equals(apiGroupPath, path)) {
            return true;
          }
        }
      }
    }
    return false;
  }
}
