/**
 * Copyright (C) 2018 Alauda
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.alauda.devops.client;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import io.alauda.kubernetes.client.Config;
import io.alauda.kubernetes.client.utils.URLUtils;
import io.alauda.kubernetes.client.utils.Utils;
import io.sundr.builder.annotations.Buildable;
import io.sundr.builder.annotations.BuildableReference;
import okhttp3.TlsVersion;

import java.util.Map;

@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true, allowGetters = true, allowSetters = true)
public class AlaudaDevOpsConfig extends Config {

  public static final String KUBERNETES_ALAUDA_API_VERSION_SYSTEM_PROPERTY = "kubernetes.api.version";
  public static final String KUBERNETES_ALAUDA_URL_SYTEM_PROPERTY = "kubernetes.url";
  public static final String ALAUDA_BUILD_TIMEOUT_SYSTEM_PROPERTY = "alauda.build.timeout";

  public static final Long DEFAULT_BUILD_TIMEOUT = 5 * 60 * 1000L;

  private String apiVersion = "v1alpha1";
  private String kubernetesUrl;
  private boolean disableApiGroupCheck;
  private long buildTimeout = DEFAULT_BUILD_TIMEOUT;

  //This is not meant to be used. This constructor is used only by the generated builder.
  AlaudaDevOpsConfig() {
  }

  public AlaudaDevOpsConfig(Config kubernetesConfig) {
    this(kubernetesConfig,
      getDefaultKubernetesUrl(kubernetesConfig), getDefaultAPIVersion(kubernetesConfig), DEFAULT_BUILD_TIMEOUT
    );
  }

  public AlaudaDevOpsConfig(Config kubernetesConfig, String kubernetesUrl) {
    this(kubernetesConfig,
      getDefaultKubernetesUrl(kubernetesConfig), getDefaultAPIVersion(kubernetesConfig), DEFAULT_BUILD_TIMEOUT
    );
    this.kubernetesUrl = kubernetesUrl;
  }

  @Buildable(builderPackage = "io.alauda.kubernetes.api.builder", editableEnabled = false, refs = {@BuildableReference(Config.class)})
  public AlaudaDevOpsConfig(String kubernetesUrl, String masterUrl, String apiVersion, String namespace, Boolean trustCerts, String caCertFile, String caCertData, String clientCertFile, String clientCertData, String clientKeyFile, String clientKeyData, String clientKeyAlgo, String clientKeyPassphrase, String username, String password, String oauthToken, int watchReconnectInterval, int watchReconnectLimit, int connectionTimeout, int requestTimeout, long rollingTimeout, long scaleTimeout, int loggingInterval, Integer maxConcurrentRequestsPerHost, String httpProxy, String httpsProxy, String[] noProxy, Map<Integer, String> errorMessages, String userAgent, TlsVersion[] tlsVersions, long buildTimeout, long websocketTimeout, long websocketPingInterval, String proxyUsername, String proxyPassword, String trustStoreFile, String trustStorePassphrase, String keyStoreFile, String keyStorePassphrase) {
    super(masterUrl, apiVersion, namespace, trustCerts, caCertFile, caCertData, clientCertFile, clientCertData, clientKeyFile, clientKeyData, clientKeyAlgo, clientKeyPassphrase, username, password, oauthToken, watchReconnectInterval, watchReconnectLimit, connectionTimeout, requestTimeout, rollingTimeout, scaleTimeout, loggingInterval,maxConcurrentRequestsPerHost, httpProxy, httpsProxy, noProxy, errorMessages, userAgent, tlsVersions, websocketTimeout, websocketPingInterval, proxyUsername, proxyPassword, trustStoreFile, trustStorePassphrase, keyStoreFile, keyStorePassphrase);
    this.apiVersion = apiVersion;
    this.kubernetesUrl = kubernetesUrl;
    this.buildTimeout = buildTimeout;
    if (this.kubernetesUrl == null || this.kubernetesUrl.isEmpty()) {
      this.kubernetesUrl = URLUtils.join(getMasterUrl(), "/");
    }

    if (!this.kubernetesUrl.endsWith("/")) {
      this.kubernetesUrl = this.kubernetesUrl + "/";
    }
  }

  public AlaudaDevOpsConfig(Config kubernetesConfig, String kubernetesUrl, String apiVersion, long buildTimeout) {
    this(kubernetesUrl,
      kubernetesConfig.getMasterUrl(), kubernetesConfig.getApiVersion(), kubernetesConfig.getNamespace(), kubernetesConfig.isTrustCerts(),
      kubernetesConfig.getCaCertFile(), kubernetesConfig.getCaCertData(),
      kubernetesConfig.getClientCertFile(), kubernetesConfig.getClientCertData(),
      kubernetesConfig.getClientKeyFile(), kubernetesConfig.getClientKeyData(),
      kubernetesConfig.getClientKeyAlgo(), kubernetesConfig.getClientKeyPassphrase(),
      kubernetesConfig.getUsername(), kubernetesConfig.getPassword(), kubernetesConfig.getOauthToken(),
      kubernetesConfig.getWatchReconnectInterval(), kubernetesConfig.getWatchReconnectLimit(),
      kubernetesConfig.getConnectionTimeout(), kubernetesConfig.getRequestTimeout(),
      kubernetesConfig.getRollingTimeout(), kubernetesConfig.getScaleTimeout(),
      kubernetesConfig.getLoggingInterval(),
      kubernetesConfig.getMaxConcurrentRequestsPerHost(),
      kubernetesConfig.getHttpProxy(),
      kubernetesConfig.getHttpsProxy(),
      kubernetesConfig.getNoProxy(),
      kubernetesConfig.getErrorMessages(),
      kubernetesConfig.getUserAgent(),
      kubernetesConfig.getTlsVersions(),
      buildTimeout,
      kubernetesConfig.getWebsocketTimeout(),
      kubernetesConfig.getWebsocketPingInterval(),
      kubernetesConfig.getProxyUsername(),
      kubernetesConfig.getProxyPassword(),
      kubernetesConfig.getTrustStoreFile(),
      kubernetesConfig.getTrustStorePassphrase(),
      kubernetesConfig.getKeyStoreFile(),
      kubernetesConfig.getKeyStorePassphrase()
      );
  }

  public static AlaudaDevOpsConfig wrap(Config config) {
    return config instanceof AlaudaDevOpsConfig ? (AlaudaDevOpsConfig) config : new AlaudaDevOpsConfig(config);
  }

  public boolean isAlaudaAPIGroups(AlaudaDevOpsClient alaudaDevOpsClient) {
    if (isDisableApiGroupCheck()) {
      return false;
    }
    return AlaudaDevOpsAdapterSupport.isAlaudaAPIGroups(alaudaDevOpsClient);
  }

  private static String getDefaultAPIVersion(Config config) {
    return Utils.getSystemPropertyOrEnvVar(KUBERNETES_ALAUDA_API_VERSION_SYSTEM_PROPERTY, config.getApiVersion());
  }

  private static String getDefaultKubernetesUrl(Config config) {
    String kubernetesUrl = Utils.getSystemPropertyOrEnvVar(KUBERNETES_ALAUDA_URL_SYTEM_PROPERTY);
    if (kubernetesUrl != null) {
      return kubernetesUrl;
    } else {
      return URLUtils.join(config.getMasterUrl(), "/");
    }
  }

//  private static boolean isRootURL(String url) {
//    try {
//      String path = new URL(url).getPath();
//      return "".equals(path) || "/".equals(path);
//    } catch (MalformedURLException e) {
//      return false;
//    }
//  }

  @JsonProperty("apiVersion")
  public String getAlaudaAPIVersion() {
    return apiVersion;
  }

  public void setOapiVersion(String apiVersion) {
    this.apiVersion = apiVersion;
  }

  @JsonProperty("kubernetesUrl")
  public String getKubernetesUrl() {
    return kubernetesUrl;
  }

  public void setKubernetesUrl(String kubernetesUrl) {
    this.kubernetesUrl = kubernetesUrl;
  }

  @JsonProperty("buildTimeout")
  public long getBuildTimeout() {
    return buildTimeout;
  }

  public void setBuildTimeout(long buildTimeout) {
    this.buildTimeout = buildTimeout;
  }

  public boolean isDisableApiGroupCheck() {
    return disableApiGroupCheck;
  }

  public void setDisableApiGroupCheck(boolean disableApiGroupCheck) {
    this.disableApiGroupCheck = disableApiGroupCheck;
  }

}
