package io.alauda.devops.java.client.extend.controller.builder;

import io.alauda.devops.java.client.extend.controller.Controllers;
import io.alauda.devops.java.client.extend.controller.DefaultControllerWatch;
import io.alauda.devops.java.client.extend.controller.reconciler.Request;
import io.alauda.devops.java.client.extend.workqueue.WorkQueue;

import java.util.function.BiPredicate;
import java.util.function.Function;
import java.util.function.Predicate;

public class ControllerWatchBuilder<ApiType> {

    private Function<ApiType, Request> workKeyGenerator;
    private WorkQueue<Request> workQueue;
    private Class<ApiType> apiTypeClass;

    private Predicate<ApiType> onAddFilterPredicate;
    private BiPredicate<ApiType, ApiType> onUpdateFilterPredicate;
    private BiPredicate<ApiType, Boolean> onDeleteFilterPredicate;

    ControllerWatchBuilder(Class<ApiType> apiTypeClass) {
        this.workQueue = null; // required, or throw an IllegalStateException in the end.
        this.apiTypeClass = apiTypeClass;
        this.workKeyGenerator = Controllers.defaultReflectiveKeyFunc();
    }

    /**
     * Sets a filter for add notification.
     *
     * @param filter the filter
     * @return the controller builder . controller watch builder
     */
    public ControllerWatchBuilder<ApiType> withOnAddFilter(Predicate<ApiType> filter) {
        this.onAddFilterPredicate = filter;
        return this;
    }

    /**
     * Sets a filter for update notification.
     *
     * @param filter the filter
     * @return the controller builder . controller watch builder
     */
    public ControllerWatchBuilder<ApiType> withOnUpdateFilter(BiPredicate<ApiType, ApiType> filter) {
        this.onUpdateFilterPredicate = filter;
        return this;
    }

    /**
     * Sets a filter for delete notification.
     *
     * @param filter the filter
     * @return the controller builder . controller watch builder
     */
    public ControllerWatchBuilder<ApiType> withOnDeleteFilter(BiPredicate<ApiType, Boolean> filter) {
        this.onDeleteFilterPredicate = filter;
        return this;
    }

    /**
     * Overrides work-queue key-func for the watch.
     *
     * @param workKeyGenerator the work key generator
     * @return the controller builder . controller watch builder
     */
    public ControllerWatchBuilder<ApiType> withWorkQueueKeyFunc(
            Function<ApiType, Request> workKeyGenerator) {
        this.workKeyGenerator = workKeyGenerator;
        return this;
    }

    public ControllerWatchBuilder<ApiType> withWorkQueue(WorkQueue<Request> workQueue) {
        this.workQueue = workQueue;
        return this;
    }

    /**
     * End building controller-watch.
     *
     * @return the controller builder
     * @throws IllegalStateException the illegal state exception
     */
    public DefaultControllerWatch<ApiType> build() throws IllegalStateException {
        // TODO: configurable resync period
        if (null == this.workQueue) {
            throw new IllegalStateException(
                    "work-queue must not be null when building work-queue event handler..");
        }
        DefaultControllerWatch<ApiType> workQueueHandler =
                new DefaultControllerWatch<>(apiTypeClass, workQueue, workKeyGenerator);
        if (onAddFilterPredicate != null) {
            workQueueHandler.setOnAddFilterPredicate(onAddFilterPredicate);
        }
        if (onUpdateFilterPredicate != null) {
            workQueueHandler.setOnUpdateFilterPredicate(onUpdateFilterPredicate);
        }
        if (onDeleteFilterPredicate != null) {
            workQueueHandler.setOnDeleteFilterPredicate(onDeleteFilterPredicate);
        }
        return workQueueHandler;
    }
}