/*
 * Decompiled with CFR 0.152.
 */
package io.ably.lib.http;

import com.google.gson.JsonParseException;
import io.ably.lib.debug.DebugOptions;
import io.ably.lib.http.HttpAuth;
import io.ably.lib.rest.Auth;
import io.ably.lib.transport.Defaults;
import io.ably.lib.transport.Hosts;
import io.ably.lib.types.AblyException;
import io.ably.lib.types.ClientOptions;
import io.ably.lib.types.ErrorInfo;
import io.ably.lib.types.ErrorResponse;
import io.ably.lib.types.Param;
import io.ably.lib.types.ProxyOptions;
import io.ably.lib.util.Log;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.Field;
import java.net.HttpURLConnection;
import java.net.InetSocketAddress;
import java.net.Proxy;
import java.net.URL;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class HttpCore {
    final String scheme;
    final int port;
    final ClientOptions options;
    final Hosts hosts;
    private final Auth auth;
    private final ProxyOptions proxyOptions;
    private HttpAuth proxyAuth;
    private Proxy proxy = Proxy.NO_PROXY;
    private boolean isDisposed;
    private static final String TAG;

    public HttpCore(ClientOptions options, Auth auth) throws AblyException {
        this.options = options;
        this.auth = auth;
        this.scheme = options.tls ? "https://" : "http://";
        this.port = Defaults.getPort(options);
        this.hosts = new Hosts(options.restHost, "rest.ably.io", options);
        this.proxyOptions = options.proxy;
        if (this.proxyOptions != null) {
            String proxyHost = this.proxyOptions.host;
            if (proxyHost == null) {
                throw AblyException.fromErrorInfo(new ErrorInfo("Unable to configure proxy without proxy host", 40000, 400));
            }
            int proxyPort = this.proxyOptions.port;
            if (proxyPort == 0) {
                throw AblyException.fromErrorInfo(new ErrorInfo("Unable to configure proxy without proxy port", 40000, 400));
            }
            this.proxy = new Proxy(Proxy.Type.HTTP, new InetSocketAddress(proxyHost, proxyPort));
            String proxyUser = this.proxyOptions.username;
            if (proxyUser != null) {
                String proxyPassword = this.proxyOptions.password;
                if (proxyPassword == null) {
                    throw AblyException.fromErrorInfo(new ErrorInfo("Unable to configure proxy without proxy password", 40000, 400));
                }
                this.proxyAuth = new HttpAuth(proxyUser, proxyPassword, this.proxyOptions.prefAuthType);
            }
        }
    }

    public <T> T httpExecuteWithRetry(URL url, String method, Param[] headers, RequestBody requestBody, ResponseHandler<T> responseHandler, boolean requireAblyAuth) throws AblyException {
        boolean renewPending = true;
        boolean proxyAuthPending = true;
        if (requireAblyAuth) {
            this.authorize(false);
        }
        while (true) {
            try {
                return this.httpExecute(url, this.getProxy(url), method, headers, requestBody, true, responseHandler);
            }
            catch (AuthRequiredException are) {
                if (are.authChallenge != null && requireAblyAuth && are.expired && renewPending) {
                    this.authorize(true);
                    renewPending = false;
                    continue;
                }
                if (are.proxyAuthChallenge != null && proxyAuthPending && this.proxyAuth != null) {
                    this.proxyAuth.processAuthenticateHeaders(are.proxyAuthChallenge);
                    proxyAuthPending = false;
                    continue;
                }
                throw are;
            }
            break;
        }
    }

    public void setHost(String host) {
        this.hosts.setHost(host);
    }

    public String getHost() {
        return this.hosts.getHost();
    }

    void authorize(boolean renew) throws AblyException {
        this.auth.assertAuthorizationHeader(renew);
    }

    synchronized void dispose() {
        if (!this.isDisposed) {
            this.isDisposed = true;
        }
    }

    public void finalize() {
        this.dispose();
    }

    public <T> T httpExecute(URL url, Proxy proxy, String method, Param[] headers, RequestBody requestBody, boolean withCredentials, ResponseHandler<T> responseHandler) throws AblyException {
        HttpURLConnection conn = null;
        try {
            conn = (HttpURLConnection)url.openConnection(proxy);
            boolean withProxyCredentials = proxy != Proxy.NO_PROXY && this.proxyAuth != null;
            T t = this.httpExecute(conn, method, headers, requestBody, withCredentials, withProxyCredentials, responseHandler);
            return t;
        }
        catch (IOException ioe) {
            throw AblyException.fromThrowable(ioe);
        }
        finally {
            if (conn != null) {
                conn.disconnect();
            }
        }
    }

    /*
     * WARNING - void declaration
     */
    <T> T httpExecute(HttpURLConnection conn, String method, Param[] headers, RequestBody requestBody, boolean withCredentials, boolean withProxyCredentials, ResponseHandler<T> responseHandler) throws AblyException {
        Response response;
        boolean credentialsIncluded = false;
        DebugOptions.RawHttpListener rawHttpListener = null;
        String id = null;
        try {
            String authHeader;
            conn.setRequestMethod(method);
            conn.setConnectTimeout(this.options.httpOpenTimeout);
            conn.setReadTimeout(this.options.httpRequestTimeout);
            conn.setDoInput(true);
            String string = authHeader = this.auth != null ? this.auth.getAuthorizationHeader() : null;
            if (withCredentials && authHeader != null) {
                conn.setRequestProperty("Authorization", authHeader);
                credentialsIncluded = true;
            }
            if (withProxyCredentials && this.proxyAuth.hasChallenge()) {
                byte[] encodedRequestBody = requestBody != null ? requestBody.getEncoded() : null;
                String string2 = this.proxyAuth.getAuthorizationHeader(method, conn.getURL().getPath(), encodedRequestBody);
                conn.setRequestProperty("Proxy-Authorization", string2);
            }
            boolean acceptSet = false;
            if (headers != null) {
                for (Param header : headers) {
                    conn.setRequestProperty(header.key, header.value);
                    if (!header.key.equals("Accept")) continue;
                    acceptSet = true;
                }
            }
            if (!acceptSet) {
                conn.setRequestProperty("Accept", "application/json");
            }
            conn.setRequestProperty("X-Ably-Version", "1.0");
            conn.setRequestProperty("X-Ably-Lib", Defaults.ABLY_LIB_VERSION);
            Object var14_18 = null;
            if (requestBody != null) {
                byte[] byArray = this.prepareRequestBody(requestBody, conn);
                if (Log.level <= 2) {
                    Log.v(TAG, System.lineSeparator() + new String(byArray));
                }
            }
            Map<String, List<String>> requestProperties = conn.getRequestProperties();
            if (Log.level <= 2) {
                Log.v(TAG, "HTTP request: " + conn.getURL() + " " + method);
                if (credentialsIncluded) {
                    Log.v(TAG, "  Authorization: " + authHeader);
                }
                for (Map.Entry<String, List<String>> entry : requestProperties.entrySet()) {
                    for (String val : entry.getValue()) {
                        Log.v(TAG, "  " + entry.getKey() + ": " + val);
                    }
                }
            }
            if (this.options instanceof DebugOptions && (rawHttpListener = ((DebugOptions)this.options).httpListener) != null) {
                id = String.valueOf(Math.random()).substring(2);
                rawHttpListener.onRawHttpRequest(id, conn, method, credentialsIncluded ? authHeader : null, requestProperties, requestBody);
            }
            if (requestBody != null) {
                void var14_20;
                this.writeRequestBody((byte[])var14_20, conn);
            }
            response = this.readResponse(conn);
            if (rawHttpListener != null) {
                rawHttpListener.onRawHttpResponse(id, response);
            }
        }
        catch (IOException ioe) {
            if (rawHttpListener != null) {
                rawHttpListener.onRawHttpException(id, ioe);
            }
            throw AblyException.fromThrowable(ioe);
        }
        return this.handleResponse(conn, credentialsIncluded, response, responseHandler);
    }

    private <T> T handleResponse(HttpURLConnection conn, boolean credentialsIncluded, Response response, ResponseHandler<T> responseHandler) throws AblyException {
        if (response.statusCode == 0) {
            return null;
        }
        if (response.statusCode >= 500 && response.statusCode <= 504) {
            ErrorInfo error = ErrorInfo.fromResponseStatus(response.statusLine, response.statusCode);
            throw AblyException.fromErrorInfo(error);
        }
        if (response.statusCode < 200 || response.statusCode >= 300) {
            List<String> proxyAuthHeaders;
            ErrorInfo error = null;
            if (response.body != null && response.body.length > 0) {
                String bodyText = new String(response.body);
                try {
                    ErrorResponse errorResponse = ErrorResponse.fromJSON(bodyText);
                    if (errorResponse != null) {
                        error = errorResponse.error;
                    }
                }
                catch (JsonParseException jse) {
                    System.err.println("Error message in unexpected format: " + bodyText);
                }
            }
            if (error == null) {
                String errorCodeHeader = conn.getHeaderField("X-Ably-ErrorCode");
                String errorMessageHeader = conn.getHeaderField("X-Ably-ErrorMessage");
                if (errorCodeHeader != null) {
                    try {
                        error = new ErrorInfo(errorMessageHeader, response.statusCode, Integer.parseInt(errorCodeHeader));
                    }
                    catch (NumberFormatException numberFormatException) {
                        // empty catch block
                    }
                }
            }
            if (response.statusCode == 401) {
                boolean stale = error != null && error.code == 40140;
                List<String> wwwAuthHeaders = response.getHeaderFields("WWW-Authenticate");
                if (wwwAuthHeaders != null && wwwAuthHeaders.size() > 0) {
                    Map<HttpAuth.Type, String> headersByType = HttpAuth.sortAuthenticateHeaders(wwwAuthHeaders);
                    String tokenHeader = headersByType.get((Object)HttpAuth.Type.X_ABLY_TOKEN);
                    if (tokenHeader != null) {
                        stale |= tokenHeader.indexOf("stale") > -1;
                    }
                    AuthRequiredException exception = new AuthRequiredException(null, error);
                    exception.authChallenge = headersByType;
                    if (stale) {
                        exception.expired = true;
                        throw exception;
                    }
                    if (!credentialsIncluded) {
                        throw exception;
                    }
                }
            }
            if (response.statusCode == 407 && (proxyAuthHeaders = response.getHeaderFields("Proxy-Authenticate")) != null && proxyAuthHeaders.size() > 0) {
                AuthRequiredException exception = new AuthRequiredException(null, error);
                exception.proxyAuthChallenge = HttpAuth.sortAuthenticateHeaders(proxyAuthHeaders);
                throw exception;
            }
            if (error == null) {
                Log.e(TAG, "Error response from server: statusCode = " + response.statusCode + "; statusLine = " + response.statusLine);
                error = ErrorInfo.fromResponseStatus(response.statusLine, response.statusCode);
            } else {
                Log.e(TAG, "Error response from server: " + error);
            }
            if (responseHandler != null) {
                return responseHandler.handleResponse(response, error);
            }
            throw AblyException.fromErrorInfo(error);
        }
        if (responseHandler != null) {
            return responseHandler.handleResponse(response, null);
        }
        return null;
    }

    private byte[] prepareRequestBody(RequestBody requestBody, HttpURLConnection conn) throws IOException {
        conn.setDoOutput(true);
        byte[] body = requestBody.getEncoded();
        int length = body.length;
        conn.setFixedLengthStreamingMode(length);
        conn.setRequestProperty("Content-Type", requestBody.getContentType());
        conn.setRequestProperty("Content-Length", Integer.toString(length));
        return body;
    }

    private void writeRequestBody(byte[] body, HttpURLConnection conn) throws IOException {
        OutputStream os = conn.getOutputStream();
        os.write(body);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Response readResponse(HttpURLConnection connection) throws IOException {
        Response response = new Response();
        response.statusCode = connection.getResponseCode();
        response.statusLine = connection.getResponseMessage();
        Log.v(TAG, "HTTP response:");
        Map<String, List<String>> caseSensitiveHeaders = connection.getHeaderFields();
        response.headers = new HashMap<String, List<String>>(caseSensitiveHeaders.size(), 1.0f);
        for (Map.Entry<String, List<String>> entry : caseSensitiveHeaders.entrySet()) {
            if (entry.getKey() == null) continue;
            response.headers.put(entry.getKey().toLowerCase(), entry.getValue());
            if (Log.level > 2) continue;
            for (String val : entry.getValue()) {
                Log.v(TAG, entry.getKey() + ": " + val);
            }
        }
        if (response.statusCode == 204) {
            return response;
        }
        response.contentType = connection.getContentType();
        response.contentLength = connection.getContentLength();
        InputStream is = null;
        try {
            is = connection.getInputStream();
        }
        catch (Throwable throwable) {
            // empty catch block
        }
        if (is == null) {
            is = connection.getErrorStream();
        }
        try {
            response.body = this.readInputStream(is, response.contentLength);
            Log.v(TAG, System.lineSeparator() + new String(response.body));
        }
        catch (NullPointerException nullPointerException) {
        }
        finally {
            if (is != null) {
                try {
                    is.close();
                }
                catch (IOException iOException) {}
            }
        }
        return response;
    }

    private byte[] readInputStream(InputStream inputStream, int bytes) throws IOException {
        if (inputStream == null) {
            throw new NullPointerException("inputStream == null");
        }
        int bytesRead = 0;
        if (bytes == -1) {
            ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
            byte[] buffer = new byte[4096];
            while ((bytesRead = inputStream.read(buffer)) > -1) {
                outputStream.write(buffer, 0, bytesRead);
            }
            return outputStream.toByteArray();
        }
        int idx = 0;
        byte[] output = new byte[bytes];
        while ((bytesRead = inputStream.read(output, idx, bytes - idx)) > -1) {
            idx += bytesRead;
        }
        return output;
    }

    Proxy getProxy(URL url) {
        String host = url.getHost();
        return this.getProxy(host);
    }

    private Proxy getProxy(String host) {
        String[] nonProxyHosts;
        if (this.proxyOptions != null && (nonProxyHosts = this.proxyOptions.nonProxyHosts) != null) {
            for (String nonProxyHostPattern : nonProxyHosts) {
                if (!host.matches(nonProxyHostPattern)) continue;
                return null;
            }
        }
        return this.proxy;
    }

    static {
        Field androidVersionField = null;
        int androidVersion = 0;
        try {
            androidVersionField = Class.forName("android.os.Build$VERSION").getField("SDK_INT");
            androidVersion = androidVersionField.getInt(androidVersionField);
        }
        catch (Exception exception) {
            // empty catch block
        }
        if (androidVersionField != null && androidVersion < 8) {
            System.setProperty("httpCore.keepAlive", "false");
        }
        TAG = HttpCore.class.getName();
    }

    public static class AuthRequiredException
    extends AblyException {
        private static final long serialVersionUID = 1L;
        public boolean expired;
        public Map<HttpAuth.Type, String> authChallenge;
        public Map<HttpAuth.Type, String> proxyAuthChallenge;

        public AuthRequiredException(Throwable throwable, ErrorInfo reason) {
            super(throwable, reason);
        }
    }

    public static class Response {
        public int statusCode;
        public String statusLine;
        public Map<String, List<String>> headers;
        public String contentType;
        public int contentLength;
        public byte[] body;

        public List<String> getHeaderFields(String name) {
            if (this.headers == null) {
                return null;
            }
            return this.headers.get(name.toLowerCase());
        }
    }

    public static interface ResponseHandler<T> {
        public T handleResponse(Response var1, ErrorInfo var2) throws AblyException;
    }

    public static interface BodyHandler<T> {
        public T[] handleResponseBody(String var1, byte[] var2) throws AblyException;
    }

    public static interface RequestBody {
        public byte[] getEncoded();

        public String getContentType();
    }
}

