/*
 * Decompiled with CFR 0.152.
 */
package io.ably.lib.http;

import com.google.gson.JsonParseException;
import io.ably.lib.http.HttpAuth;
import io.ably.lib.http.HttpUtils;
import io.ably.lib.rest.Auth;
import io.ably.lib.transport.Defaults;
import io.ably.lib.transport.Hosts;
import io.ably.lib.types.AblyException;
import io.ably.lib.types.ClientOptions;
import io.ably.lib.types.ErrorInfo;
import io.ably.lib.types.ErrorResponse;
import io.ably.lib.types.Param;
import io.ably.lib.types.ProxyOptions;
import io.ably.lib.util.Base64Coder;
import io.ably.lib.util.Log;
import io.ably.lib.util.Serialisation;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.Field;
import java.net.HttpURLConnection;
import java.net.InetSocketAddress;
import java.net.MalformedURLException;
import java.net.Proxy;
import java.net.URL;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class Http {
    public static final String GET = "GET";
    public static final String POST = "POST";
    public static final String DELETE = "DELETE";
    final String scheme;
    final int port;
    final ClientOptions options;
    final Hosts hosts;
    private final Auth auth;
    private String authHeader;
    private final ProxyOptions proxyOptions;
    private HttpAuth proxyAuth;
    private Proxy proxy = Proxy.NO_PROXY;
    private boolean isDisposed;
    private static final String TAG;
    private static final String LINK = "Link";
    private static final String ACCEPT = "Accept";
    private static final String CONTENT_TYPE = "Content-Type";
    private static final String CONTENT_LENGTH = "Content-Length";
    private static final String JSON = "application/json";
    private static final String WWW_AUTHENTICATE = "WWW-Authenticate";
    private static final String PROXY_AUTHENTICATE = "Proxy-Authenticate";
    private static final String AUTHORIZATION = "Authorization";
    private static final String PROXY_AUTHORIZATION = "Proxy-Authorization";

    public Http(ClientOptions options, Auth auth) throws AblyException {
        this.options = options;
        this.auth = auth;
        this.scheme = options.tls ? "https://" : "http://";
        this.port = Defaults.getPort(options);
        this.hosts = new Hosts(options.restHost, "rest.ably.io", options);
        this.proxyOptions = options.proxy;
        if (this.proxyOptions != null) {
            String proxyHost = this.proxyOptions.host;
            if (proxyHost == null) {
                throw AblyException.fromErrorInfo(new ErrorInfo("Unable to configure proxy without proxy host", 40000, 400));
            }
            int proxyPort = this.proxyOptions.port;
            if (proxyPort == 0) {
                throw AblyException.fromErrorInfo(new ErrorInfo("Unable to configure proxy without proxy port", 40000, 400));
            }
            this.proxy = new Proxy(Proxy.Type.HTTP, new InetSocketAddress(proxyHost, proxyPort));
            String proxyUser = this.proxyOptions.username;
            if (proxyUser != null) {
                String proxyPassword = this.proxyOptions.password;
                if (proxyPassword == null) {
                    throw AblyException.fromErrorInfo(new ErrorInfo("Unable to configure proxy without proxy password", 40000, 400));
                }
                this.proxyAuth = new HttpAuth(proxyUser, proxyPassword, this.proxyOptions.prefAuthType);
            }
        }
    }

    public void setHost(String host) {
        this.hosts.setHost(host);
    }

    public String getHost() {
        return this.hosts.getHost();
    }

    public String getUrlString(String url) throws AblyException {
        return new String(this.getUrl(url));
    }

    public byte[] getUrl(String url) throws AblyException {
        try {
            return this.httpExecute(new URL(url), GET, null, null, new ResponseHandler<byte[]>(){

                @Override
                public byte[] handleResponse(int statusCode, String contentType, Collection<String> linkHeaders, byte[] body) throws AblyException {
                    return body;
                }
            });
        }
        catch (IOException ioe) {
            throw AblyException.fromThrowable(ioe);
        }
    }

    public <T> T getUri(String uri, Param[] headers, Param[] params, ResponseHandler<T> responseHandler) throws AblyException {
        return this.httpExecute(Http.buildURL(uri, params), GET, headers, null, responseHandler);
    }

    public <T> T get(String path, Param[] headers, Param[] params, ResponseHandler<T> responseHandler) throws AblyException {
        return this.ablyHttpExecute(path, GET, headers, params, null, responseHandler);
    }

    public <T> T post(String path, Param[] headers, Param[] params, RequestBody requestBody, ResponseHandler<T> responseHandler) throws AblyException {
        return this.ablyHttpExecute(path, POST, headers, params, requestBody, responseHandler);
    }

    public <T> T del(String path, Param[] headers, Param[] params, ResponseHandler<T> responseHandler) throws AblyException {
        return this.ablyHttpExecute(path, DELETE, headers, params, null, responseHandler);
    }

    private String getAuthorizationHeader(boolean renew) throws AblyException {
        if (this.authHeader != null && !renew) {
            return this.authHeader;
        }
        if (this.auth.getAuthMethod() == Auth.AuthMethod.basic) {
            this.authHeader = "Basic " + Base64Coder.encodeString(this.auth.getBasicCredentials());
        } else {
            Auth.AuthOptions options = null;
            if (renew) {
                options = new Auth.AuthOptions();
                options.force = true;
            }
            this.auth.authorise(options, null);
            this.authHeader = "Bearer " + this.auth.getTokenAuth().getEncodedToken();
        }
        return this.authHeader;
    }

    void authorise(boolean renew) throws AblyException {
        this.getAuthorizationHeader(renew);
    }

    synchronized void dispose() {
        if (!this.isDisposed) {
            this.isDisposed = true;
        }
    }

    public void finalize() {
        this.dispose();
    }

    public <T> T ablyHttpExecute(String path, String method, Param[] headers, Param[] params, RequestBody requestBody, ResponseHandler<T> responseHandler) throws AblyException {
        String candidateHost = this.getHost();
        int retryCountRemaining = this.hosts.getFallback(candidateHost) != null ? this.options.httpMaxRetryCount : 0;
        while (true) {
            URL url = Http.buildURL(this.scheme, candidateHost, this.port, path, params);
            try {
                return this.httpExecuteWithRetry(url, method, headers, requestBody, responseHandler, true);
            }
            catch (AblyException.HostFailedException e) {
                if (--retryCountRemaining < 0) {
                    throw e;
                }
                Log.d(TAG, "Connection failed to host `" + candidateHost + "`. Searching for new host...");
                candidateHost = this.hosts.getFallback(candidateHost);
                if (candidateHost == null) {
                    throw e;
                }
                Log.d(TAG, "Switched to `" + candidateHost + "`.");
                continue;
            }
            break;
        }
    }

    public <T> T httpExecute(URL url, String method, Param[] headers, RequestBody requestBody, ResponseHandler<T> responseHandler) throws AblyException {
        return this.httpExecuteWithRetry(url, method, headers, requestBody, responseHandler, false);
    }

    public <T> T httpExecute(URL url, Proxy proxy, String method, Param[] headers, RequestBody requestBody, boolean withCredentials, ResponseHandler<T> responseHandler) throws AblyException {
        HttpURLConnection conn = null;
        try {
            conn = (HttpURLConnection)url.openConnection(proxy);
            boolean withProxyCredentials = proxy != Proxy.NO_PROXY && this.proxyAuth != null;
            T t = this.httpExecute(conn, method, headers, requestBody, withCredentials, withProxyCredentials, responseHandler);
            return t;
        }
        catch (IOException ioe) {
            throw AblyException.fromThrowable(ioe);
        }
        finally {
            if (conn != null) {
                conn.disconnect();
            }
        }
    }

    <T> T httpExecute(HttpURLConnection conn, String method, Param[] headers, RequestBody requestBody, boolean withCredentials, boolean withProxyCredentials, ResponseHandler<T> responseHandler) throws AblyException {
        Response response;
        boolean credentialsIncluded = false;
        try {
            conn.setRequestMethod(method);
            conn.setConnectTimeout(this.options.httpOpenTimeout);
            conn.setReadTimeout(this.options.httpRequestTimeout);
            conn.setDoInput(true);
            if (withCredentials && this.authHeader != null) {
                conn.setRequestProperty(AUTHORIZATION, this.authHeader);
                credentialsIncluded = true;
            }
            if (withProxyCredentials && this.proxyAuth.hasChallenge()) {
                byte[] encodedRequestBody = requestBody != null ? requestBody.getEncoded() : null;
                String proxyAuthorizationHeader = this.proxyAuth.getAuthorizationHeader(method, conn.getURL().getPath(), encodedRequestBody);
                conn.setRequestProperty(PROXY_AUTHORIZATION, proxyAuthorizationHeader);
            }
            boolean acceptSet = false;
            if (headers != null) {
                for (Param header : headers) {
                    conn.setRequestProperty(header.key, header.value);
                    if (!header.key.equals(ACCEPT)) continue;
                    acceptSet = true;
                }
            }
            if (!acceptSet) {
                conn.setRequestProperty(ACCEPT, JSON);
            }
            conn.setRequestProperty("X-Ably-Version", "0.8");
            conn.setRequestProperty("X-Ably-Lib", HttpUtils.X_ABLY_LIB_VALUE);
            if (requestBody != null) {
                this.writeRequestBody(requestBody, conn);
            }
            response = this.readResponse(conn);
        }
        catch (IOException ioe) {
            throw AblyException.fromThrowable(ioe);
        }
        return this.handleResponse(conn, credentialsIncluded, response, responseHandler);
    }

    public <T> T httpExecuteWithRetry(URL url, String method, Param[] headers, RequestBody requestBody, ResponseHandler<T> responseHandler, boolean allowAblyAuth) throws AblyException {
        boolean authPending = true;
        boolean renewPending = true;
        boolean proxyAuthPending = true;
        while (true) {
            try {
                return this.httpExecute(url, this.getProxy(url), method, headers, requestBody, true, responseHandler);
            }
            catch (AuthRequiredException are) {
                if (are.authChallenge != null && allowAblyAuth) {
                    if (authPending) {
                        this.authorise(false);
                        authPending = false;
                        continue;
                    }
                    if (are.expired && renewPending) {
                        this.authorise(true);
                        renewPending = false;
                        continue;
                    }
                }
                if (are.proxyAuthChallenge != null && proxyAuthPending && this.proxyAuth != null) {
                    this.proxyAuth.processAuthenticateHeaders(are.proxyAuthChallenge);
                    proxyAuthPending = false;
                    continue;
                }
                throw are;
            }
            break;
        }
    }

    private <T> T handleResponse(HttpURLConnection conn, boolean credentialsIncluded, Response response, ResponseHandler<T> responseHandler) throws AblyException {
        if (response.statusCode == 0) {
            return null;
        }
        if (response.statusCode >= 500 && response.statusCode <= 504) {
            throw AblyException.fromErrorInfo(ErrorInfo.fromResponseStatus(response.statusLine, response.statusCode));
        }
        if (response.statusCode < 200 || response.statusCode >= 300) {
            List<String> proxyAuthHeaders;
            ErrorInfo error = null;
            if (response.body != null && response.body.length > 0) {
                String bodyText = new String(response.body);
                try {
                    ErrorResponse errorResponse = ErrorResponse.fromJSON(bodyText);
                    if (errorResponse != null) {
                        error = errorResponse.error;
                    }
                }
                catch (JsonParseException jse) {
                    System.err.println("Error message in unexpected format: " + bodyText);
                }
            }
            if (error == null) {
                String errorCodeHeader = conn.getHeaderField("X-Ably-ErrorCode");
                String errorMessageHeader = conn.getHeaderField("X-Ably-ErrorMessage");
                if (errorCodeHeader != null) {
                    try {
                        error = new ErrorInfo(errorMessageHeader, response.statusCode, Integer.parseInt(errorCodeHeader));
                    }
                    catch (NumberFormatException numberFormatException) {
                        // empty catch block
                    }
                }
            }
            if (response.statusCode == 401) {
                boolean stale = error != null && error.code == 40140;
                List<String> wwwAuthHeaders = response.getHeaderFields(WWW_AUTHENTICATE);
                if (wwwAuthHeaders != null && wwwAuthHeaders.size() > 0) {
                    Map<HttpAuth.Type, String> headersByType = HttpAuth.sortAuthenticateHeaders(wwwAuthHeaders);
                    String tokenHeader = headersByType.get((Object)HttpAuth.Type.X_ABLY_TOKEN);
                    if (tokenHeader != null) {
                        stale |= tokenHeader.indexOf("stale") > -1;
                    }
                    AuthRequiredException exception = new AuthRequiredException(null, error);
                    exception.authChallenge = headersByType;
                    if (stale) {
                        exception.expired = true;
                        throw exception;
                    }
                    if (!credentialsIncluded) {
                        throw exception;
                    }
                }
            }
            if (response.statusCode == 407 && (proxyAuthHeaders = response.getHeaderFields(PROXY_AUTHENTICATE)) != null && proxyAuthHeaders.size() > 0) {
                AuthRequiredException exception = new AuthRequiredException(null, error);
                exception.proxyAuthChallenge = HttpAuth.sortAuthenticateHeaders(proxyAuthHeaders);
                throw exception;
            }
            if (error != null) {
                Log.e(TAG, "Error response from server: " + error);
                throw AblyException.fromErrorInfo(error);
            }
            Log.e(TAG, "Error response from server: statusCode = " + response.statusCode + "; statusLine = " + response.statusLine);
            throw AblyException.fromErrorInfo(ErrorInfo.fromResponseStatus(response.statusLine, response.statusCode));
        }
        if (responseHandler == null) {
            return null;
        }
        List<String> linkHeaders = response.getHeaderFields(LINK);
        return responseHandler.handleResponse(response.statusCode, response.contentType, linkHeaders, response.body);
    }

    private void writeRequestBody(RequestBody requestBody, HttpURLConnection conn) throws IOException {
        conn.setDoOutput(true);
        byte[] body = requestBody.getEncoded();
        int length = body.length;
        conn.setFixedLengthStreamingMode(length);
        conn.setRequestProperty(CONTENT_TYPE, requestBody.getContentType());
        conn.setRequestProperty(CONTENT_LENGTH, Integer.toString(length));
        OutputStream os = conn.getOutputStream();
        os.write(body);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Response readResponse(HttpURLConnection connection) throws IOException {
        Response response = new Response();
        response.statusCode = connection.getResponseCode();
        response.statusLine = connection.getResponseMessage();
        Map<String, List<String>> caseSensitiveHeaders = connection.getHeaderFields();
        response.headers = new HashMap<String, List<String>>(caseSensitiveHeaders.size(), 1.0f);
        for (Map.Entry<String, List<String>> entry : caseSensitiveHeaders.entrySet()) {
            if (entry.getKey() == null) continue;
            response.headers.put(entry.getKey().toLowerCase(), entry.getValue());
        }
        if (response.statusCode == 204) {
            return response;
        }
        response.contentType = connection.getContentType();
        response.contentLength = connection.getContentLength();
        int successStatusCode = POST.equals(connection.getRequestMethod()) ? 201 : 200;
        InputStream is = response.statusCode == successStatusCode ? connection.getInputStream() : connection.getErrorStream();
        try {
            response.body = this.readInputStream(is, response.contentLength);
        }
        catch (NullPointerException nullPointerException) {
        }
        finally {
            if (is != null) {
                try {
                    is.close();
                }
                catch (IOException iOException) {}
            }
        }
        return response;
    }

    private byte[] readInputStream(InputStream inputStream, int bytes) throws IOException {
        if (inputStream == null) {
            throw new NullPointerException("inputStream == null");
        }
        int bytesRead = 0;
        if (bytes == -1) {
            ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
            byte[] buffer = new byte[4096];
            while ((bytesRead = inputStream.read(buffer)) > -1) {
                outputStream.write(buffer, 0, bytesRead);
            }
            return outputStream.toByteArray();
        }
        int idx = 0;
        byte[] output = new byte[bytes];
        while ((bytesRead = inputStream.read(output, idx, bytes - idx)) > -1) {
            idx += bytesRead;
        }
        return output;
    }

    private static void appendParams(StringBuilder uri, Param[] params) {
        if (params != null && params.length > 0) {
            uri.append('?').append(params[0].key).append('=').append(params[0].value);
            for (int i = 1; i < params.length; ++i) {
                uri.append('&').append(params[i].key).append('=').append(params[i].value);
            }
        }
    }

    static URL buildURL(String scheme, String host, int port, String path, Param[] params) {
        StringBuilder builder = new StringBuilder(scheme).append(host).append(':').append(port).append(path);
        Http.appendParams(builder, params);
        URL result = null;
        try {
            result = new URL(builder.toString());
        }
        catch (MalformedURLException malformedURLException) {
            // empty catch block
        }
        return result;
    }

    static URL buildURL(String uri, Param[] params) {
        StringBuilder builder = new StringBuilder(uri);
        Http.appendParams(builder, params);
        URL result = null;
        try {
            result = new URL(builder.toString());
        }
        catch (MalformedURLException malformedURLException) {
            // empty catch block
        }
        return result;
    }

    Proxy getProxy(URL url) {
        String host = url.getHost();
        return this.getProxy(host);
    }

    private Proxy getProxy(String host) {
        String[] nonProxyHosts;
        if (this.proxyOptions != null && (nonProxyHosts = this.proxyOptions.nonProxyHosts) != null) {
            for (String nonProxyHostPattern : nonProxyHosts) {
                if (!host.matches(nonProxyHostPattern)) continue;
                return null;
            }
        }
        return this.proxy;
    }

    static {
        Field androidVersionField = null;
        int androidVersion = 0;
        try {
            androidVersionField = Class.forName("android.os.Build$VERSION").getField("SDK_INT");
            androidVersion = androidVersionField.getInt(androidVersionField);
        }
        catch (Exception exception) {
            // empty catch block
        }
        if (androidVersionField != null && androidVersion < 8) {
            System.setProperty("http.keepAlive", "false");
        }
        TAG = Http.class.getName();
    }

    public static class ByteArrayRequestBody
    implements RequestBody {
        private final byte[] bytes;
        private final String contentType;

        public ByteArrayRequestBody(byte[] bytes, String contentType) {
            this.bytes = bytes;
            this.contentType = contentType;
        }

        @Override
        public byte[] getEncoded() {
            return this.bytes;
        }

        @Override
        public String getContentType() {
            return this.contentType;
        }
    }

    public static class JSONRequestBody
    implements RequestBody {
        private final String jsonText;
        private byte[] bytes;

        public JSONRequestBody(String jsonText) {
            this.jsonText = jsonText;
        }

        public JSONRequestBody(Object ob) {
            this(Serialisation.gson.toJson(ob));
        }

        @Override
        public byte[] getEncoded() {
            return this.bytes != null ? this.bytes : (this.bytes = this.jsonText.getBytes());
        }

        @Override
        public String getContentType() {
            return Http.JSON;
        }
    }

    private static class Response {
        int statusCode;
        String statusLine;
        Map<String, List<String>> headers;
        String contentType;
        int contentLength;
        byte[] body;

        private Response() {
        }

        public List<String> getHeaderFields(String name) {
            if (this.headers == null) {
                return null;
            }
            return this.headers.get(name.toLowerCase());
        }
    }

    public static class AuthRequiredException
    extends AblyException {
        private static final long serialVersionUID = 1L;
        public boolean expired;
        public Map<HttpAuth.Type, String> authChallenge;
        public Map<HttpAuth.Type, String> proxyAuthChallenge;

        public AuthRequiredException(Throwable throwable, ErrorInfo reason) {
            super(throwable, reason);
        }
    }

    public static interface RequestBody {
        public byte[] getEncoded();

        public String getContentType();
    }

    public static interface BodyHandler<T> {
        public T[] handleResponseBody(String var1, byte[] var2) throws AblyException;
    }

    public static interface ResponseHandler<T> {
        public T handleResponse(int var1, String var2, Collection<String> var3, byte[] var4) throws AblyException;
    }
}

