/*
 * Decompiled with CFR 0.152.
 */
package io.ably.lib.http;

import io.ably.lib.rest.Auth;
import io.ably.lib.transport.Defaults;
import io.ably.lib.transport.Hosts;
import io.ably.lib.types.AblyException;
import io.ably.lib.types.ClientOptions;
import io.ably.lib.types.ErrorInfo;
import io.ably.lib.types.ErrorResponse;
import io.ably.lib.types.Param;
import io.ably.lib.util.Base64Coder;
import io.ably.lib.util.Log;
import io.ably.lib.util.Serialisation;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.Field;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class Http {
    private final ClientOptions options;
    private final Auth auth;
    private final String scheme;
    private String host;
    private final int port;
    private String authHeader;
    private boolean isDisposed;
    private static final String TAG;
    private static final String LINK = "Link";
    private static final String ACCEPT = "Accept";
    private static final String CONTENT_TYPE = "Content-Type";
    private static final String CONTENT_LENGTH = "Content-Length";
    private static final String JSON = "application/json";
    private static final String WWW_AUTHENTICATE = "WWW-Authenticate";
    private static final String AUTHORIZATION = "Authorization";
    static final String GET = "GET";
    static final String POST = "POST";
    static final String DELETE = "DELETE";

    public Http(ClientOptions options, Auth auth) {
        this.options = options;
        this.auth = auth;
        this.host = options.restHost;
        this.scheme = options.tls ? "https://" : "http://";
        this.port = Defaults.getPort(options);
    }

    public void setHost(String host) {
        this.host = host;
    }

    public String getHost() {
        return this.host;
    }

    public String getUrlString(String url) throws AblyException {
        return new String(this.getUrl(url));
    }

    public byte[] getUrl(String url) throws AblyException {
        try {
            return this.httpExecute(new URL(url), GET, null, null, false, new ResponseHandler<byte[]>(){

                @Override
                public byte[] handleResponse(int statusCode, String contentType, Collection<String> linkHeaders, byte[] body) throws AblyException {
                    return body;
                }
            });
        }
        catch (IOException ioe) {
            throw AblyException.fromThrowable(ioe);
        }
    }

    public <T> T getUri(String uri, Param[] headers, Param[] params, ResponseHandler<T> responseHandler) throws AblyException {
        return this.httpExecute(this.buildURL(uri, params), GET, headers, null, false, responseHandler);
    }

    public <T> T get(String path, Param[] headers, Param[] params, ResponseHandler<T> responseHandler) throws AblyException {
        return this.ablyHttpExecute(path, GET, headers, params, null, responseHandler);
    }

    public <T> T post(String path, Param[] headers, Param[] params, RequestBody requestBody, ResponseHandler<T> responseHandler) throws AblyException {
        return this.ablyHttpExecute(path, POST, headers, params, requestBody, responseHandler);
    }

    public <T> T del(String path, Param[] headers, Param[] params, ResponseHandler<T> responseHandler) throws AblyException {
        return this.ablyHttpExecute(path, DELETE, headers, params, null, responseHandler);
    }

    private String getAuthorizationHeader(boolean renew) throws AblyException {
        if (this.authHeader != null && !renew) {
            return this.authHeader;
        }
        if (this.auth.getAuthMethod() == Auth.AuthMethod.basic) {
            this.authHeader = "Basic " + Base64Coder.encodeString(this.auth.getBasicCredentials());
        } else {
            Auth.AuthOptions options = null;
            if (renew) {
                options = new Auth.AuthOptions();
                options.force = true;
            }
            this.auth.authorise(options, null);
            this.authHeader = "Bearer " + this.auth.getTokenAuth().getEncodedToken();
        }
        return this.authHeader;
    }

    private void authorise(boolean renew) throws AblyException {
        this.getAuthorizationHeader(renew);
    }

    synchronized void dispose() {
        if (!this.isDisposed) {
            this.isDisposed = true;
        }
    }

    public void finalize() {
        this.dispose();
    }

    <T> T ablyHttpExecute(String path, String method, Param[] headers, Param[] params, RequestBody requestBody, ResponseHandler<T> responseHandler) throws AblyException {
        int retryCountRemaining = Hosts.isRestFallbackSupported(this.host) ? this.options.httpMaxRetryCount : 0;
        String hostCurrent = this.host;
        while (true) {
            URL url = this.buildURL(this.scheme, hostCurrent, path, params);
            try {
                return this.httpExecute(url, method, headers, requestBody, true, responseHandler);
            }
            catch (AblyException.HostFailedException e) {
                if (--retryCountRemaining < 0) continue;
                Log.d(TAG, "Connection failed to host `" + hostCurrent + "`. Searching for new host...");
                hostCurrent = Hosts.getFallback(hostCurrent);
                Log.d(TAG, "Switched to `" + hostCurrent + "`.");
                if (retryCountRemaining >= 0) continue;
                throw AblyException.fromErrorInfo(new ErrorInfo("Connection failed; no host available", 404, 80000));
            }
            break;
        }
    }

    <T> T httpExecute(URL url, String method, Param[] headers, RequestBody requestBody, boolean withCredentials, ResponseHandler<T> responseHandler) throws AblyException {
        Response response;
        HttpURLConnection conn = null;
        boolean credentialsIncluded = false;
        try {
            conn = (HttpURLConnection)url.openConnection();
            conn.setConnectTimeout(this.options.httpOpenTimeout);
            conn.setReadTimeout(this.options.httpRequestTimeout);
            conn.setDoInput(true);
            if (method != null) {
                conn.setRequestMethod(method);
            }
            if (withCredentials && this.authHeader != null) {
                credentialsIncluded = true;
                conn.setRequestProperty(AUTHORIZATION, this.authHeader);
            }
            boolean acceptSet = false;
            if (headers != null) {
                for (Param header : headers) {
                    conn.setRequestProperty(header.key, header.value);
                    if (!header.key.equals(ACCEPT)) continue;
                    acceptSet = true;
                }
            }
            if (!acceptSet) {
                conn.setRequestProperty(ACCEPT, JSON);
            }
            if (requestBody != null) {
                this.writeRequestBody(requestBody, conn);
            }
            response = this.readResponse(conn);
        }
        catch (IOException ioe) {
            throw AblyException.fromThrowable(ioe);
        }
        finally {
            if (conn != null) {
                conn.disconnect();
            }
        }
        if (response.statusCode == 0) {
            return null;
        }
        if (response.statusCode >= 500 && response.statusCode <= 504) {
            throw AblyException.fromErrorInfo(ErrorInfo.fromResponseStatus(response.statusLine, response.statusCode));
        }
        if (response.statusCode < 200 || response.statusCode >= 300) {
            String wwwAuthHeader;
            ErrorResponse errorResponse;
            ErrorInfo error = null;
            if (response.body != null && response.body.length > 0 && (errorResponse = ErrorResponse.fromJSON(new String(response.body))) != null) {
                error = errorResponse.error;
            }
            if (error == null) {
                String errorCodeHeader = conn.getHeaderField("X-Ably-ErrorCode");
                String errorMessageHeader = conn.getHeaderField("X-Ably-ErrorMessage");
                if (errorCodeHeader != null) {
                    try {
                        error = new ErrorInfo(errorMessageHeader, response.statusCode, Integer.parseInt(errorCodeHeader));
                    }
                    catch (NumberFormatException numberFormatException) {
                        // empty catch block
                    }
                }
            }
            if (response.statusCode == 401 && (wwwAuthHeader = response.getHeaderField(WWW_AUTHENTICATE)) != null) {
                boolean stale;
                boolean bl = stale = wwwAuthHeader.indexOf("stale") > -1 || error != null && error.code == 40140;
                if (withCredentials && (stale || !credentialsIncluded)) {
                    this.authorise(stale);
                    return this.httpExecute(url, method, headers, requestBody, withCredentials, responseHandler);
                }
            }
            if (error != null) {
                Log.e(TAG, "Error response from server: " + error);
                throw AblyException.fromErrorInfo(error);
            }
            Log.e(TAG, "Error response from server: statusCode = " + response.statusCode + "; statusLine = " + response.statusLine);
            throw AblyException.fromErrorInfo(ErrorInfo.fromResponseStatus(response.statusLine, response.statusCode));
        }
        if (responseHandler == null) {
            return null;
        }
        List<String> linkHeaders = response.getHeaderFields(LINK);
        return responseHandler.handleResponse(response.statusCode, response.contentType, linkHeaders, response.body);
    }

    private void writeRequestBody(RequestBody requestBody, HttpURLConnection conn) throws IOException {
        conn.setDoOutput(true);
        byte[] body = requestBody.getEncoded();
        int length = body.length;
        conn.setFixedLengthStreamingMode(length);
        conn.setRequestProperty(CONTENT_TYPE, requestBody.getContentType());
        conn.setRequestProperty(CONTENT_LENGTH, Integer.toString(length));
        OutputStream os = conn.getOutputStream();
        os.write(body);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Response readResponse(HttpURLConnection connection) throws IOException {
        Response response = new Response();
        response.statusCode = connection.getResponseCode();
        response.statusLine = connection.getResponseMessage();
        Map<String, List<String>> caseSensitiveHeaders = connection.getHeaderFields();
        response.headers = new HashMap<String, List<String>>(caseSensitiveHeaders.size(), 1.0f);
        for (Map.Entry<String, List<String>> entry : caseSensitiveHeaders.entrySet()) {
            if (entry.getKey() == null) continue;
            response.headers.put(entry.getKey().toLowerCase(), entry.getValue());
        }
        if (response.statusCode == 204) {
            return response;
        }
        response.contentType = connection.getContentType();
        response.contentLength = connection.getContentLength();
        int successStatusCode = POST.equals(connection.getRequestMethod()) ? 201 : 200;
        InputStream is = response.statusCode == successStatusCode ? connection.getInputStream() : connection.getErrorStream();
        try {
            response.body = this.readInputStream(is, response.contentLength);
        }
        catch (NullPointerException nullPointerException) {
        }
        finally {
            if (is != null) {
                try {
                    is.close();
                }
                catch (IOException iOException) {}
            }
        }
        return response;
    }

    private byte[] readInputStream(InputStream inputStream, int bytes) throws IOException {
        if (inputStream == null) {
            throw new NullPointerException("inputStream == null");
        }
        int bytesRead = 0;
        if (bytes == -1) {
            ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
            byte[] buffer = new byte[4096];
            while ((bytesRead = inputStream.read(buffer)) > -1) {
                outputStream.write(buffer, 0, bytesRead);
            }
            return outputStream.toByteArray();
        }
        int idx = 0;
        byte[] output = new byte[bytes];
        while ((bytesRead = inputStream.read(output, idx, bytes - idx)) > -1) {
            idx += bytesRead;
        }
        return output;
    }

    private void appendParams(StringBuilder uri, Param[] params) {
        if (params != null && params.length > 0) {
            uri.append('?').append(params[0].key).append('=').append(params[0].value);
            for (int i = 1; i < params.length; ++i) {
                uri.append('&').append(params[i].key).append('=').append(params[i].value);
            }
        }
    }

    private URL buildURL(String scheme, String host, String path, Param[] params) {
        StringBuilder builder = new StringBuilder(scheme).append(host).append(':').append(this.port).append(path);
        this.appendParams(builder, params);
        URL result = null;
        try {
            result = new URL(builder.toString());
        }
        catch (MalformedURLException malformedURLException) {
            // empty catch block
        }
        return result;
    }

    private URL buildURL(String uri, Param[] params) {
        StringBuilder builder = new StringBuilder(uri);
        this.appendParams(builder, params);
        URL result = null;
        try {
            result = new URL(builder.toString());
        }
        catch (MalformedURLException malformedURLException) {
            // empty catch block
        }
        return result;
    }

    static {
        Field androidVersionField = null;
        int androidVersion = 0;
        try {
            androidVersionField = Class.forName("android.os.Build$VERSION").getField("SDK_INT");
            androidVersion = androidVersionField.getInt(androidVersionField);
        }
        catch (Exception exception) {
            // empty catch block
        }
        if (androidVersionField != null && androidVersion < 8) {
            System.setProperty("http.keepAlive", "false");
        }
        TAG = Http.class.getName();
    }

    public static class ByteArrayRequestBody
    implements RequestBody {
        private final byte[] bytes;
        private final String contentType;

        public ByteArrayRequestBody(byte[] bytes, String contentType) {
            this.bytes = bytes;
            this.contentType = contentType;
        }

        @Override
        public byte[] getEncoded() {
            return this.bytes;
        }

        @Override
        public String getContentType() {
            return this.contentType;
        }
    }

    public static class JSONRequestBody
    implements RequestBody {
        private final String jsonText;

        public JSONRequestBody(String jsonText) {
            this.jsonText = jsonText;
        }

        public JSONRequestBody(Object ob) {
            this(Serialisation.gson.toJson(ob));
        }

        @Override
        public byte[] getEncoded() {
            return this.jsonText.getBytes();
        }

        @Override
        public String getContentType() {
            return Http.JSON;
        }
    }

    private static class Response {
        int statusCode;
        String statusLine;
        Map<String, List<String>> headers;
        String contentType;
        int contentLength;
        byte[] body;

        private Response() {
        }

        public String getHeaderField(String name) {
            List<String> fields = this.getHeaderFields(name);
            if (fields == null || fields.isEmpty()) {
                return null;
            }
            return fields.get(fields.size() - 1);
        }

        public List<String> getHeaderFields(String name) {
            if (this.headers == null) {
                return null;
            }
            return this.headers.get(name.toLowerCase());
        }
    }

    public static interface RequestBody {
        public byte[] getEncoded();

        public String getContentType();
    }

    public static interface BodyHandler<T> {
        public T[] handleResponseBody(String var1, byte[] var2) throws AblyException;
    }

    public static interface ResponseHandler<T> {
        public T handleResponse(int var1, String var2, Collection<String> var3, byte[] var4) throws AblyException;
    }
}

