/*
 *
 * ====================================================================
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *  ====================================================================
 */

/*
 * Changes to the original project are Copyright 2019, Verizon Media Inc..
 */

package dev.amp.validator.visitor;

import dev.amp.validator.ValidatorProtos;
import dev.amp.validator.css.CssTokenUtil;
import dev.amp.validator.css.QualifiedRule;
import dev.amp.validator.css.Token;
import dev.amp.validator.css.ErrorToken;
import dev.amp.validator.css.AtRule;
import dev.amp.validator.css.CssValidationException;

import javax.annotation.Nonnull;
import java.util.ArrayList;
import java.util.List;

/**
 * A visitor for keyframes at rule.
 *
 * @author nhant01
 * @author GeorgeLuo
 */

public class KeyframesVisitor implements RuleVisitor {
    /**
     * Constructor.
     *
     * @param errors the errors collection to populate
     */
    public KeyframesVisitor(@Nonnull final List<ErrorToken> errors) {
        super();
        this.errors = errors;
        this.parentIsKeyframesAtRule = false;
    }

    /**
     * visit a qualified rule
     *
     * @param qualifiedRule to visit
     * @throws CssValidationException Css Validation Error
     */
    @Override
    public void visitQualifiedRule(@Nonnull final QualifiedRule qualifiedRule) throws CssValidationException {
        if (!this.parentIsKeyframesAtRule) {
            List<String> params = new ArrayList<>();
            params.add("style");
            params.add(qualifiedRule.ruleName());
            this.errors.add(createErrorTokenAt(
                    qualifiedRule,
                    ValidatorProtos.ValidationError.Code.CSS_SYNTAX_DISALLOWED_QUALIFIED_RULE_MUST_BE_INSIDE_KEYFRAME,
                    params));
            return;
        }
        if (qualifiedRule.getDeclarations().size() > 0) {
            return;
        }
        List<String> params = new ArrayList<>();
        params.add("style");
        params.add(qualifiedRule.ruleName());
        this.errors.add(createErrorTokenAt(
                qualifiedRule,
                ValidatorProtos.ValidationError.Code.CSS_SYNTAX_QUALIFIED_RULE_HAS_NO_DECLARATIONS,
                params));
    }


    /**
     * Fills an ErrorToken with the provided position, code, and params.
     *
     * @param positionToken generating error
     * @param code          the code generated by the error
     * @param params        the error params
     * @return an ErrorToken positioned at positionToken location
     * @throws CssValidationException Css Validation Error
     */
    public static ErrorToken createErrorTokenAt(@Nonnull final Token positionToken,
                                                @Nonnull final ValidatorProtos.ValidationError.Code code,
                                                @Nonnull final List<String> params) throws CssValidationException {
        ErrorToken token = new ErrorToken(code, params);
        CssTokenUtil.copyPosTo(positionToken, token);
        return token;
    }

    /**
     * Visit an atRule.
     *x
     * @param atRule to visit
     * @throws CssValidationException Css Validation Error
     */
    @Override
    public void visitAtRule(@Nonnull final AtRule atRule) throws CssValidationException {
        switch (atRule.getName()) {
            case "keyframes":
            case "-moz-keyframes":
            case "-o-keyframes":
            case "-webkit-keyframes":
                if (this.parentIsKeyframesAtRule) {
                    List<String> params = new ArrayList<>();
                    params.add("style");
                    this.errors.add(createErrorTokenAt(
                            atRule,
                            ValidatorProtos.ValidationError.Code.CSS_SYNTAX_DISALLOWED_KEYFRAME_INSIDE_KEYFRAME,
                            params));
                }
                this.parentIsKeyframesAtRule = true;
                return;
            default:
        }
    }

    /**
     * Leave an atRule.
     *
     * @param atRule currently visiting
     */
    @Override
    public void leaveAtRule(@Nonnull final AtRule atRule) {
        this.parentIsKeyframesAtRule = false;
    }

    /** List of token errors. */
    @Nonnull
    private final List<ErrorToken> errors;

    /** Flag to indicate whether if the parent is key frame at rule. */
    private boolean parentIsKeyframesAtRule;
}
