/*
 * Decompiled with CFR 0.152.
 */
package com.hubspot.singularity.executor;

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import com.google.inject.name.Named;
import com.hubspot.mesos.JavaUtils;
import com.hubspot.singularity.SingularityTaskExecutorData;
import com.hubspot.singularity.executor.SingularityExecutorCgroupCfsChecker;
import com.hubspot.singularity.executor.SingularityExecutorProcessKiller;
import com.hubspot.singularity.executor.SingularityExecutorThreadChecker;
import com.hubspot.singularity.executor.config.SingularityExecutorConfiguration;
import com.hubspot.singularity.executor.config.SingularityExecutorLogging;
import com.hubspot.singularity.executor.task.SingularityExecutorTask;
import com.hubspot.singularity.executor.task.SingularityExecutorTaskProcessCallable;
import com.hubspot.singularity.executor.utils.ExecutorUtils;
import com.hubspot.singularity.runner.base.shared.WatchServiceHelper;
import java.util.Collection;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.Callable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.mesos.ExecutorDriver;
import org.apache.mesos.Protos;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class SingularityExecutorMonitor {
    private static final Logger LOG = LoggerFactory.getLogger(SingularityExecutorMonitor.class);
    private final ListeningExecutorService processBuilderPool;
    private final ListeningExecutorService runningProcessPool;
    private final ScheduledExecutorService exitChecker;
    private final ExecutorService cgroupCfsWatcherService;
    private final Lock exitLock;
    private final AtomicBoolean alreadyShutDown;
    private final CountDownLatch latch;
    private volatile Optional<Future> exitCheckerFuture;
    private volatile RunState runState;
    private final SingularityExecutorConfiguration configuration;
    private final SingularityExecutorLogging logging;
    private final ExecutorUtils executorUtils;
    private final SingularityExecutorProcessKiller processKiller;
    private final SingularityExecutorThreadChecker threadChecker;
    private final Map<String, SingularityExecutorTask> tasks;
    private final Map<String, ListenableFuture<ProcessBuilder>> processBuildingTasks;
    private final Map<String, SingularityExecutorTaskProcessCallable> processRunningTasks;
    private final Map<String, ListeningExecutorService> taskToShellCommandPool;
    private final Map<String, SingularityExecutorCgroupCfsChecker> cgroupCheckers;

    @Inject
    public SingularityExecutorMonitor(@Named(value="already.shut.down") AtomicBoolean alreadyShutDown, SingularityExecutorLogging logging, ExecutorUtils executorUtils, SingularityExecutorProcessKiller processKiller, SingularityExecutorThreadChecker threadChecker, SingularityExecutorConfiguration configuration) {
        this.logging = logging;
        this.configuration = configuration;
        this.executorUtils = executorUtils;
        this.processKiller = processKiller;
        this.exitChecker = Executors.newSingleThreadScheduledExecutor(new ThreadFactoryBuilder().setNameFormat("SingularityExecutorExitChecker-%d").build());
        this.threadChecker = threadChecker;
        this.threadChecker.start(this);
        this.tasks = Maps.newConcurrentMap();
        this.processBuildingTasks = Maps.newConcurrentMap();
        this.processRunningTasks = Maps.newConcurrentMap();
        this.taskToShellCommandPool = Maps.newConcurrentMap();
        this.cgroupCheckers = Maps.newConcurrentMap();
        this.processBuilderPool = MoreExecutors.listeningDecorator((ExecutorService)Executors.newCachedThreadPool(new ThreadFactoryBuilder().setNameFormat("SingularityExecutorProcessBuilder-%d").build()));
        this.runningProcessPool = MoreExecutors.listeningDecorator((ExecutorService)Executors.newCachedThreadPool(new ThreadFactoryBuilder().setNameFormat("SingularityExecutorProcessRunner-%d").build()));
        this.cgroupCfsWatcherService = Executors.newSingleThreadExecutor(new ThreadFactoryBuilder().setNameFormat("cgroup-cfs-watcher-%d").build());
        this.runState = RunState.STARTING;
        this.exitLock = new ReentrantLock();
        this.alreadyShutDown = alreadyShutDown;
        this.latch = new CountDownLatch(4);
    }

    public void start(ExecutorDriver driver) {
        Preconditions.checkState((this.runState == RunState.STARTING ? 1 : 0) != 0);
        this.runState = RunState.RUNNING;
        this.exitCheckerFuture = Optional.of(this.startExitChecker(driver, this.configuration.getInitialIdleExecutorShutdownWaitMillis()));
    }

    public void shutdown(ExecutorDriver driver) {
        if (!this.alreadyShutDown.compareAndSet(false, true)) {
            LOG.info("Already ran shut down process");
            return;
        }
        LOG.info("Shutdown requested with driver {}", (Object)driver);
        this.threadChecker.getExecutorService().shutdown();
        this.processBuilderPool.shutdown();
        this.runningProcessPool.shutdown();
        this.cgroupCfsWatcherService.shutdown();
        for (SingularityExecutorTask singularityExecutorTask : this.tasks.values()) {
            if (singularityExecutorTask.wasKilled()) continue;
            singularityExecutorTask.getLog().info("Executor shutting down - requested task kill with state: {}", (Object)this.requestKill(singularityExecutorTask.getTaskId()));
        }
        this.processKiller.getExecutorService().shutdown();
        for (Map.Entry entry : this.taskToShellCommandPool.entrySet()) {
            LOG.warn("Shutting down abandoned pool for {}", entry.getKey());
            ((ListeningExecutorService)entry.getValue()).shutdown();
        }
        this.cgroupCheckers.values().forEach(WatchServiceHelper::close);
        this.exitChecker.shutdown();
        long start = System.currentTimeMillis();
        JavaUtils.awaitTerminationWithLatch((CountDownLatch)this.latch, (String)"threadChecker", (ExecutorService)this.threadChecker.getExecutorService(), (long)this.configuration.getShutdownTimeoutWaitMillis());
        JavaUtils.awaitTerminationWithLatch((CountDownLatch)this.latch, (String)"processBuilder", (ExecutorService)this.processBuilderPool, (long)this.configuration.getShutdownTimeoutWaitMillis());
        JavaUtils.awaitTerminationWithLatch((CountDownLatch)this.latch, (String)"runningProcess", (ExecutorService)this.runningProcessPool, (long)this.configuration.getShutdownTimeoutWaitMillis());
        JavaUtils.awaitTerminationWithLatch((CountDownLatch)this.latch, (String)"processKiller", (ExecutorService)this.processKiller.getExecutorService(), (long)this.configuration.getShutdownTimeoutWaitMillis());
        LOG.info("Awaiting shutdown of all thread pools for a max of {}", (Object)JavaUtils.durationFromMillis((long)this.configuration.getShutdownTimeoutWaitMillis()));
        try {
            this.latch.await();
        }
        catch (InterruptedException e) {
            LOG.warn("While awaiting shutdown of executor services", (Throwable)e);
        }
        LOG.info("Waited {} for shutdown of thread pools, now waiting {} before exiting...", (Object)JavaUtils.duration((long)start), (Object)JavaUtils.durationFromMillis((long)this.configuration.getStopDriverAfterMillis()));
        try {
            Thread.sleep(this.configuration.getStopDriverAfterMillis());
        }
        catch (Throwable t) {
            LOG.warn("While waiting to exit", t);
        }
        LOG.info("Stopping driver {}", (Object)driver);
        Protos.Status status = driver.stop();
        LOG.info("Driver stopped with status {}", (Object)status);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void checkForExit(ExecutorDriver driver, long waitMillis) {
        try {
            this.exitLock.lockInterruptibly();
        }
        catch (InterruptedException e) {
            LOG.warn("Interrupted acquiring exit lock", (Throwable)e);
            return;
        }
        boolean shuttingDown = false;
        try {
            if (this.tasks.isEmpty()) {
                LOG.info("Shutting down executor due to no tasks being submitted within {}", (Object)JavaUtils.durationFromMillis((long)waitMillis));
                this.runState = RunState.SHUTDOWN;
                shuttingDown = true;
            }
        }
        finally {
            this.exitLock.unlock();
        }
        if (shuttingDown) {
            this.shutdown(driver);
        } else if (this.runState == RunState.SHUTDOWN) {
            LOG.info("Already shutting down...");
        } else {
            LOG.info("Tasks wasn't empty, exit checker doing nothing...");
        }
    }

    private Future startExitChecker(final ExecutorDriver driver, final long waitTimeMillis) {
        LOG.info("Starting an exit checker that will run in {}", (Object)JavaUtils.durationFromMillis((long)waitTimeMillis));
        return this.exitChecker.schedule(new Runnable(){

            @Override
            public void run() {
                LOG.info("Exit checker running...");
                try {
                    SingularityExecutorMonitor.this.checkForExit(driver, waitTimeMillis);
                }
                catch (Throwable t) {
                    SingularityExecutorMonitor.this.logAndExit(2, "While shutting down", new Object[]{t});
                }
            }
        }, waitTimeMillis, TimeUnit.MILLISECONDS);
    }

    private void clearExitCheckerUnsafe() {
        if (this.exitCheckerFuture.isPresent()) {
            LOG.info("Canceling an exit checker");
            this.exitCheckerFuture.get().cancel(true);
            this.exitCheckerFuture = Optional.empty();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public SubmitState submit(SingularityExecutorTask task) {
        this.exitLock.lock();
        try {
            ReentrantLock taskLock = task.getLock();
            taskLock.lock();
            try {
                if (this.runState == RunState.SHUTDOWN) {
                    this.finishTask(task, Protos.TaskState.TASK_LOST, "Task couldn't start because executor is shutting down", Optional.empty(), new Object[0]);
                    SubmitState submitState = SubmitState.REJECTED;
                    return submitState;
                }
                if (this.tasks.containsKey(task.getTaskId())) {
                    SubmitState submitState = SubmitState.TASK_ALREADY_EXISTED;
                    return submitState;
                }
                this.tasks.put(task.getTaskId(), task);
                this.clearExitCheckerUnsafe();
                ListenableFuture processBuildFuture = this.processBuilderPool.submit((Callable)task.getProcessBuilder());
                this.processBuildingTasks.put(task.getTaskId(), (ListenableFuture<ProcessBuilder>)processBuildFuture);
                this.watchProcessBuilder(task, (ListenableFuture<ProcessBuilder>)processBuildFuture);
            }
            finally {
                taskLock.unlock();
            }
        }
        finally {
            this.exitLock.unlock();
        }
        return SubmitState.SUBMITTED;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void logAndExit(int statusCode, String format, Object ... args) {
        try {
            LOG.error(format, args);
        }
        finally {
            System.exit(statusCode);
        }
    }

    public Collection<SingularityExecutorTaskProcessCallable> getRunningTasks() {
        return this.processRunningTasks.values();
    }

    public Optional<SingularityExecutorTaskProcessCallable> getTaskProcess(String taskId) {
        return Optional.ofNullable(this.processRunningTasks.get(taskId));
    }

    public Optional<SingularityExecutorTask> getTask(String taskId) {
        return Optional.ofNullable(this.tasks.get(taskId));
    }

    public ListeningExecutorService getShellCommandExecutorServiceForTask(String taskId) {
        if (!this.taskToShellCommandPool.containsKey(taskId)) {
            ListeningExecutorService executorService = MoreExecutors.listeningDecorator((ExecutorService)Executors.newCachedThreadPool(new ThreadFactoryBuilder().setNameFormat(taskId + "-shellCommandPool-%d").build()));
            this.taskToShellCommandPool.put(taskId, executorService);
        }
        return this.taskToShellCommandPool.get(taskId);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void finishTask(SingularityExecutorTask task, Protos.TaskState taskState, String message, Optional<String> errorMsg, Object ... errorObjects) {
        block6: {
            try {
                if (!errorMsg.isPresent()) break block6;
                task.getLog().error(errorMsg.get(), errorObjects);
            }
            catch (Throwable throwable) {
                try {
                    this.sendStatusUpdate(task, taskState, message);
                    this.onFinish(task, taskState);
                }
                catch (Throwable t) {
                    this.logAndExit(3, "Failed while finishing task {} (state {})", task.getTaskId(), taskState, t);
                }
                throw throwable;
            }
        }
        try {
            this.sendStatusUpdate(task, taskState, message);
            this.onFinish(task, taskState);
        }
        catch (Throwable t) {
            this.logAndExit(3, "Failed while finishing task {} (state {})", task.getTaskId(), taskState, t);
        }
    }

    private void watchProcessBuilder(final SingularityExecutorTask task, ListenableFuture<ProcessBuilder> processBuildFuture) {
        Futures.addCallback(processBuildFuture, (FutureCallback)new FutureCallback<ProcessBuilder>(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            private void onSuccessThrows(ProcessBuilder processBuilder) {
                task.getLog().debug("Process builder finished succesfully... ");
                boolean wasKilled = false;
                ReentrantLock taskLock = task.getLock();
                taskLock.lock();
                try {
                    SingularityExecutorMonitor.this.processBuildingTasks.remove(task.getTaskId());
                    wasKilled = task.wasKilled();
                    if (!wasKilled) {
                        SingularityExecutorMonitor.this.processRunningTasks.put(task.getTaskId(), SingularityExecutorMonitor.this.submitProcessMonitor(task, processBuilder));
                        SingularityExecutorMonitor.this.startCgroupWatcher(task);
                    }
                }
                finally {
                    taskLock.unlock();
                }
                if (wasKilled) {
                    SingularityExecutorMonitor.this.finishTask(task, Protos.TaskState.TASK_KILLED, "Task killed before service process started", Optional.empty(), new Object[0]);
                }
            }

            public void onSuccess(ProcessBuilder processBuilder) {
                try {
                    this.onSuccessThrows(processBuilder);
                }
                catch (Throwable t) {
                    SingularityExecutorMonitor.this.finishTask(task, Protos.TaskState.TASK_LOST, String.format("Task lost while transitioning due to: %s", t.getClass().getSimpleName()), Optional.of("While submitting process task"), t);
                }
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            public void onFailure(Throwable t) {
                Protos.TaskState state = Protos.TaskState.TASK_LOST;
                String message = String.format("%s while initializing task: %s", t.getClass().getSimpleName(), t.getMessage());
                try {
                    if (task.wasKilled()) {
                        state = Protos.TaskState.TASK_KILLED;
                        message = String.format("Task killed, caught expected %s", t.getClass().getSimpleName());
                    }
                }
                catch (Throwable throwable) {
                    SingularityExecutorMonitor.this.finishTask(task, state, message, Optional.of("Task {} failed before starting process"), task, t);
                    throw throwable;
                }
                SingularityExecutorMonitor.this.finishTask(task, state, message, Optional.of("Task {} failed before starting process"), task, t);
            }
        }, (Executor)this.getShellCommandExecutorServiceForTask(task.getTaskId()));
    }

    private void startCgroupWatcher(SingularityExecutorTask task) {
        SingularityTaskExecutorData taskExecutorData = (SingularityTaskExecutorData)task.getExecutorData();
        if (taskExecutorData.getCpuHardLimit().isPresent()) {
            this.cgroupCfsWatcherService.submit(() -> {
                try {
                    SingularityExecutorCgroupCfsChecker cfsChecker = new SingularityExecutorCgroupCfsChecker(task, (Integer)taskExecutorData.getCpuHardLimit().get(), this.configuration.getDefaultCfsPeriod());
                    cfsChecker.watch();
                    this.cgroupCheckers.put(task.getTaskId(), cfsChecker);
                }
                catch (Throwable t) {
                    LOG.error("Could not start cgroup checker for task {}", (Object)task.getTaskId(), (Object)t);
                }
            });
        }
    }

    private void sendStatusUpdate(SingularityExecutorTask task, Protos.TaskState taskState, String message) {
        this.executorUtils.sendStatusUpdate(task.getDriver(), Protos.TaskID.newBuilder().setValue(task.getTaskId()).build(), taskState, message, task.getLog());
    }

    private void onFinish(SingularityExecutorTask task, Protos.TaskState taskState) {
        this.processKiller.cancelDestroyFuture(task.getTaskId());
        this.tasks.remove(task.getTaskId());
        this.processRunningTasks.remove(task.getTaskId());
        this.processBuildingTasks.remove(task.getTaskId());
        task.cleanup(taskState);
        ListeningExecutorService executorService = this.taskToShellCommandPool.remove(task.getTaskId());
        if (executorService != null) {
            executorService.shutdownNow();
            try {
                executorService.awaitTermination(5L, TimeUnit.MILLISECONDS);
            }
            catch (InterruptedException e) {
                LOG.warn("Awaiting shutdown of shell executor service", (Throwable)e);
            }
        }
        this.logging.stopTaskLogger(task.getTaskId(), task.getLogbackLog());
        this.checkIdleExecutorShutdown(task.getDriver());
    }

    private void checkIdleExecutorShutdown(ExecutorDriver driver) {
        this.exitLock.lock();
        try {
            this.clearExitCheckerUnsafe();
            if (this.tasks.isEmpty() && this.runState == RunState.RUNNING) {
                this.exitCheckerFuture = Optional.of(this.startExitChecker(driver, this.configuration.getIdleExecutorShutdownWaitMillis()));
            }
        }
        finally {
            this.exitLock.unlock();
        }
    }

    public KillState requestKill(String taskId) {
        return this.requestKill(taskId, Optional.empty(), false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public KillState requestKill(String taskId, Optional<String> user, boolean destroy) {
        Optional<SingularityExecutorTask> maybeTask = Optional.ofNullable(this.tasks.get(taskId));
        if (!maybeTask.isPresent()) {
            return KillState.DIDNT_EXIST;
        }
        SingularityExecutorTask task = maybeTask.get();
        if (!destroy && task.wasForceDestroyed()) {
            task.getLog().debug("Already force destroyed, will not issue additional kill");
            return KillState.DESTROYING_PROCESS;
        }
        task.getLog().info("Executor asked to kill {}", (Object)taskId);
        ListenableFuture<ProcessBuilder> processBuilderFuture = null;
        SingularityExecutorTaskProcessCallable runningProcess = null;
        task.getLock().lock();
        boolean wasKilled = task.wasKilled();
        try {
            if (!wasKilled) {
                task.markKilled(user);
            }
            processBuilderFuture = this.processBuildingTasks.get(task.getTaskId());
            runningProcess = this.processRunningTasks.get(task.getTaskId());
        }
        finally {
            task.getLock().unlock();
        }
        if (processBuilderFuture != null) {
            task.getLog().info("Canceling process builder future for {}", (Object)taskId);
            CancelThread cancelThread = new CancelThread(processBuilderFuture, task);
            cancelThread.start();
            return KillState.INTERRUPTING_PRE_PROCESS;
        }
        if (runningProcess != null) {
            if (destroy) {
                if (user.isPresent()) {
                    task.getLog().info("Destroying process with pid {} for task {} by request from user {}", new Object[]{runningProcess.getCurrentPid(), taskId, user.get()});
                } else {
                    task.getLog().info("Destroying process with pid {} for task {}", (Object)runningProcess.getCurrentPid(), (Object)taskId);
                }
                task.markForceDestroyed(user);
                runningProcess.signalKillToProcessIfActive();
                return KillState.DESTROYING_PROCESS;
            }
            if (this.processKiller.isKillInProgress(taskId)) {
                task.getLog().info("Kill already in progress for task {}", (Object)taskId);
                return KillState.KILLING_PROCESS;
            }
            if (user.isPresent()) {
                task.getLog().info("Killing process for task {} by request from {}", (Object)taskId, (Object)user.get());
            } else {
                task.getLog().info("Killing process for task {}", (Object)taskId);
            }
            this.processKiller.submitKillRequest(runningProcess);
            return KillState.KILLING_PROCESS;
        }
        return KillState.INCONSISTENT_STATE;
    }

    private SingularityExecutorTaskProcessCallable buildProcessCallable(SingularityExecutorTask task, ProcessBuilder processBuilder) {
        return new SingularityExecutorTaskProcessCallable(this.configuration, task, processBuilder, this.executorUtils);
    }

    private SingularityExecutorTaskProcessCallable submitProcessMonitor(SingularityExecutorTask task, ProcessBuilder processBuilder) {
        SingularityExecutorTaskProcessCallable processCallable = this.buildProcessCallable(task, processBuilder);
        ListenableFuture processExitFuture = this.runningProcessPool.submit((Callable)processCallable);
        this.watchProcessExitFuture(task, (ListenableFuture<Integer>)processExitFuture);
        return processCallable;
    }

    private void watchProcessExitFuture(final SingularityExecutorTask task, ListenableFuture<Integer> processExitFuture) {
        Futures.addCallback(processExitFuture, (FutureCallback)new FutureCallback<Integer>(){

            public void onSuccess(Integer exitCode) {
                Protos.TaskState taskState = null;
                String message = null;
                Optional<String> maybeKilledBy = task.getKilledBy();
                if (task.wasKilledDueToThreads()) {
                    taskState = Protos.TaskState.TASK_FAILED;
                    message = String.format("Task used %s threads and was killed (max %s)", task.getThreadCountAtOverageTime(), task.getExecutorData().getMaxTaskThreads().get());
                } else if (task.wasKilled()) {
                    taskState = Protos.TaskState.TASK_KILLED;
                    if (task.wasDestroyedAfterWaiting()) {
                        long millisWaited = task.getExecutorData().getSigKillProcessesAfterMillis().orElse(SingularityExecutorMonitor.this.configuration.getHardKillAfterMillis());
                        message = String.format("Task killed forcibly after waiting at least %s", JavaUtils.durationFromMillis((long)millisWaited));
                    } else {
                        message = task.wasForceDestroyed() ? (maybeKilledBy.isPresent() ? String.format("Task killed forcibly by %s", maybeKilledBy.get()) : "Task killed forcibly after multiple kill requests from framework") : "Task killed. Process exited gracefully with code " + exitCode;
                    }
                } else if (task.isSuccessExitCode(exitCode)) {
                    taskState = Protos.TaskState.TASK_FINISHED;
                    message = "Process exited normally with code " + exitCode;
                } else {
                    taskState = Protos.TaskState.TASK_FAILED;
                    message = "Process failed with code " + exitCode;
                }
                SingularityExecutorMonitor.this.sendStatusUpdate(task, taskState, message);
                SingularityExecutorMonitor.this.onFinish(task, taskState);
            }

            public void onFailure(Throwable t) {
                task.getLog().error("Task {} failed while running process", (Object)task, (Object)t);
                Protos.TaskState taskState = null;
                String message = null;
                if (task.wasKilled()) {
                    taskState = Protos.TaskState.TASK_KILLED;
                    message = String.format("Task killed, caught %s", t.getClass().getSimpleName());
                } else {
                    taskState = Protos.TaskState.TASK_LOST;
                    message = String.format("%s while running process %s", t.getClass().getSimpleName(), t.getMessage());
                }
                SingularityExecutorMonitor.this.sendStatusUpdate(task, taskState, message);
                SingularityExecutorMonitor.this.onFinish(task, taskState);
            }
        }, (Executor)this.getShellCommandExecutorServiceForTask(task.getTaskId()));
    }

    private static class CancelThread
    extends Thread {
        private final ListenableFuture<ProcessBuilder> processBuilderFuture;
        private final SingularityExecutorTask task;

        public CancelThread(ListenableFuture<ProcessBuilder> processBuilderFuture, SingularityExecutorTask task) {
            super("SingularityExecutorMonitor-cancel-thread");
            this.processBuilderFuture = processBuilderFuture;
            this.task = task;
        }

        @Override
        public void run() {
            this.processBuilderFuture.cancel(true);
            this.task.getProcessBuilder().cancel();
        }
    }

    public static enum KillState {
        DIDNT_EXIST,
        INTERRUPTING_PRE_PROCESS,
        KILLING_PROCESS,
        DESTROYING_PROCESS,
        INCONSISTENT_STATE;

    }

    public static enum SubmitState {
        SUBMITTED,
        REJECTED,
        TASK_ALREADY_EXISTED;

    }

    public static enum RunState {
        STARTING,
        RUNNING,
        SHUTDOWN;

    }
}

