/*
 * Decompiled with CFR 0.152.
 */
package com.hubspot.singularity.executor;

import com.google.common.base.Optional;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.google.inject.Inject;
import com.hubspot.mesos.JavaUtils;
import com.hubspot.singularity.executor.SingularityExecutorProcessKiller;
import com.hubspot.singularity.executor.SingularityExecutorThreadChecker;
import com.hubspot.singularity.executor.config.SingularityExecutorConfiguration;
import com.hubspot.singularity.executor.config.SingularityExecutorLogging;
import com.hubspot.singularity.executor.task.SingularityExecutorTask;
import com.hubspot.singularity.executor.task.SingularityExecutorTaskProcessCallable;
import com.hubspot.singularity.executor.utils.ExecutorUtils;
import java.util.Collection;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.mesos.ExecutorDriver;
import org.apache.mesos.Protos;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SingularityExecutorMonitor {
    private static final Logger LOG = LoggerFactory.getLogger(SingularityExecutorMonitor.class);
    private final ListeningExecutorService processBuilderPool;
    private final ListeningExecutorService runningProcessPool;
    private final ScheduledExecutorService exitChecker;
    private final Lock exitLock;
    private volatile Optional<Future> exitCheckerFuture;
    private volatile RunState runState;
    private final SingularityExecutorConfiguration configuration;
    private final SingularityExecutorLogging logging;
    private final ExecutorUtils executorUtils;
    private final SingularityExecutorProcessKiller processKiller;
    private final SingularityExecutorThreadChecker threadChecker;
    private final Map<String, SingularityExecutorTask> tasks;
    private final Map<String, ListenableFuture<ProcessBuilder>> processBuildingTasks;
    private final Map<String, SingularityExecutorTaskProcessCallable> processRunningTasks;

    @Inject
    public SingularityExecutorMonitor(SingularityExecutorLogging logging, ExecutorUtils executorUtils, SingularityExecutorProcessKiller processKiller, SingularityExecutorThreadChecker threadChecker, SingularityExecutorConfiguration configuration) {
        this.logging = logging;
        this.configuration = configuration;
        this.executorUtils = executorUtils;
        this.processKiller = processKiller;
        this.exitChecker = Executors.newSingleThreadScheduledExecutor(new ThreadFactoryBuilder().setNameFormat("SingularityExecutorExitChecker-%d").build());
        this.threadChecker = threadChecker;
        this.threadChecker.start(this);
        this.tasks = Maps.newConcurrentMap();
        this.processBuildingTasks = Maps.newConcurrentMap();
        this.processRunningTasks = Maps.newConcurrentMap();
        this.processBuilderPool = MoreExecutors.listeningDecorator((ExecutorService)Executors.newCachedThreadPool(new ThreadFactoryBuilder().setNameFormat("SingularityExecutorProcessBuilder-%d").build()));
        this.runningProcessPool = MoreExecutors.listeningDecorator((ExecutorService)Executors.newCachedThreadPool(new ThreadFactoryBuilder().setNameFormat("SingularityExecutorProcessRunner-%d").build()));
        this.runState = RunState.RUNNING;
        this.exitLock = new ReentrantLock();
        this.exitCheckerFuture = Optional.of((Object)this.startExitChecker((Optional<ExecutorDriver>)Optional.absent()));
    }

    public void shutdown(Optional<ExecutorDriver> driver) {
        LOG.info("Shutdown requested with driver {}", driver);
        this.threadChecker.getExecutorService().shutdown();
        this.processBuilderPool.shutdown();
        this.runningProcessPool.shutdown();
        for (SingularityExecutorTask task : this.tasks.values()) {
            task.getLog().info("Executor shutting down - requested task kill with state: {}", (Object)this.requestKill(task.getTaskId()));
        }
        this.processKiller.getExecutorService().shutdown();
        this.exitChecker.shutdown();
        CountDownLatch latch = new CountDownLatch(4);
        JavaUtils.awaitTerminationWithLatch((CountDownLatch)latch, (String)"threadChecker", (ExecutorService)this.threadChecker.getExecutorService(), (long)this.configuration.getShutdownTimeoutWaitMillis());
        JavaUtils.awaitTerminationWithLatch((CountDownLatch)latch, (String)"processBuilder", (ExecutorService)this.processBuilderPool, (long)this.configuration.getShutdownTimeoutWaitMillis());
        JavaUtils.awaitTerminationWithLatch((CountDownLatch)latch, (String)"runningProcess", (ExecutorService)this.runningProcessPool, (long)this.configuration.getShutdownTimeoutWaitMillis());
        JavaUtils.awaitTerminationWithLatch((CountDownLatch)latch, (String)"processKiller", (ExecutorService)this.processKiller.getExecutorService(), (long)this.configuration.getShutdownTimeoutWaitMillis());
        LOG.info("Awaiting shutdown of all executor services for a max of {}", (Object)JavaUtils.durationFromMillis((long)this.configuration.getShutdownTimeoutWaitMillis()));
        try {
            latch.await();
        }
        catch (InterruptedException e) {
            LOG.warn("While awaiting shutdown of executor services", (Throwable)e);
        }
        LOG.info("Waiting {} before exiting...", (Object)JavaUtils.durationFromMillis((long)this.configuration.getStopDriverAfterMillis()));
        try {
            Thread.sleep(this.configuration.getStopDriverAfterMillis());
        }
        catch (Throwable t) {
            LOG.warn("While waiting to exit", t);
        }
        if (driver.isPresent()) {
            LOG.info("Stopping driver {}", driver.get());
            ((ExecutorDriver)driver.get()).stop();
        } else {
            this.logAndExit(1, "No driver present on shutdown, exiting", new Object[0]);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void checkForExit(Optional<ExecutorDriver> driver) {
        try {
            this.exitLock.lockInterruptibly();
        }
        catch (InterruptedException e) {
            LOG.warn("Interrupted acquiring exit lock", (Throwable)e);
            return;
        }
        boolean shuttingDown = false;
        try {
            if (this.tasks.isEmpty()) {
                this.runState = RunState.SHUTDOWN;
                shuttingDown = true;
            }
        }
        finally {
            this.exitLock.unlock();
        }
        if (shuttingDown) {
            this.shutdown(driver);
        } else if (this.runState == RunState.SHUTDOWN) {
            LOG.info("Already shutting down...");
        } else {
            LOG.info("Tasks wasn't empty, exit checker doing nothing...");
        }
    }

    private Future startExitChecker(final Optional<ExecutorDriver> driver) {
        LOG.info("Starting an exit checker that will run in {}", (Object)JavaUtils.durationFromMillis((long)this.configuration.getIdleExecutorShutdownWaitMillis()));
        return this.exitChecker.schedule(new Runnable(){

            @Override
            public void run() {
                LOG.info("Exit checker running...");
                try {
                    SingularityExecutorMonitor.this.checkForExit((Optional<ExecutorDriver>)driver);
                }
                catch (Throwable t) {
                    SingularityExecutorMonitor.this.logAndExit(2, "While shutting down", new Object[]{t});
                }
            }
        }, this.configuration.getIdleExecutorShutdownWaitMillis(), TimeUnit.MILLISECONDS);
    }

    private void clearExitCheckerUnsafe() {
        if (this.exitCheckerFuture.isPresent()) {
            LOG.info("Canceling an exit checker");
            ((Future)this.exitCheckerFuture.get()).cancel(true);
            this.exitCheckerFuture = Optional.absent();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public SubmitState submit(SingularityExecutorTask task) {
        this.exitLock.lock();
        try {
            ReentrantLock taskLock = task.getLock();
            taskLock.lock();
            try {
                if (this.runState == RunState.SHUTDOWN) {
                    this.finishTask(task, Protos.TaskState.TASK_LOST, "Task couldn't start because executor is shutting down", (Optional<String>)Optional.absent(), new Object[0]);
                    SubmitState submitState = SubmitState.REJECTED;
                    return submitState;
                }
                if (this.tasks.containsKey(task.getTaskId())) {
                    SubmitState submitState = SubmitState.TASK_ALREADY_EXISTED;
                    return submitState;
                }
                this.tasks.put(task.getTaskId(), task);
                this.clearExitCheckerUnsafe();
                ListenableFuture processBuildFuture = this.processBuilderPool.submit((Callable)task.getProcessBuilder());
                this.processBuildingTasks.put(task.getTaskId(), (ListenableFuture<ProcessBuilder>)processBuildFuture);
                this.watchProcessBuilder(task, (ListenableFuture<ProcessBuilder>)processBuildFuture);
            }
            finally {
                taskLock.unlock();
            }
        }
        finally {
            this.exitLock.unlock();
        }
        return SubmitState.SUBMITTED;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void logAndExit(int statusCode, String format, Object ... args) {
        try {
            LOG.error(format, args);
        }
        finally {
            System.exit(statusCode);
        }
    }

    public Collection<SingularityExecutorTaskProcessCallable> getRunningTasks() {
        return this.processRunningTasks.values();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void finishTask(SingularityExecutorTask task, Protos.TaskState taskState, String message, Optional<String> errorMsg, Object ... errorObjects) {
        block6: {
            try {
                this.processKiller.cancelDestroyFuture(task.getTaskId());
                if (!errorMsg.isPresent()) break block6;
                task.getLog().error((String)errorMsg.get(), errorObjects);
            }
            catch (Throwable throwable) {
                try {
                    this.sendStatusUpdate(task, taskState, message);
                    this.onFinish(task, taskState);
                }
                catch (Throwable t) {
                    this.logAndExit(3, "Failed while finishing task {} (state {})", task.getTaskId(), taskState, t);
                }
                throw throwable;
            }
        }
        try {
            this.sendStatusUpdate(task, taskState, message);
            this.onFinish(task, taskState);
        }
        catch (Throwable t) {
            this.logAndExit(3, "Failed while finishing task {} (state {})", task.getTaskId(), taskState, t);
        }
    }

    private void watchProcessBuilder(final SingularityExecutorTask task, ListenableFuture<ProcessBuilder> processBuildFuture) {
        Futures.addCallback(processBuildFuture, (FutureCallback)new FutureCallback<ProcessBuilder>(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            private void onSuccessThrows(ProcessBuilder processBuilder) {
                task.getLog().debug("Process builder finished succesfully... ");
                boolean wasKilled = false;
                ReentrantLock taskLock = task.getLock();
                taskLock.lock();
                try {
                    SingularityExecutorMonitor.this.processBuildingTasks.remove(task.getTaskId());
                    wasKilled = task.wasKilled();
                    if (!wasKilled) {
                        SingularityExecutorMonitor.this.processRunningTasks.put(task.getTaskId(), SingularityExecutorMonitor.this.submitProcessMonitor(task, processBuilder));
                    }
                }
                finally {
                    taskLock.unlock();
                }
                if (wasKilled) {
                    SingularityExecutorMonitor.this.finishTask(task, Protos.TaskState.TASK_KILLED, "Task killed before service process started", (Optional<String>)Optional.absent(), new Object[0]);
                }
            }

            public void onSuccess(ProcessBuilder processBuilder) {
                try {
                    this.onSuccessThrows(processBuilder);
                }
                catch (Throwable t) {
                    SingularityExecutorMonitor.this.finishTask(task, Protos.TaskState.TASK_LOST, String.format("Task lost while transitioning due to: %s", t.getClass().getSimpleName()), (Optional<String>)Optional.of((Object)"While submitting process task"), t);
                }
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            public void onFailure(Throwable t) {
                Protos.TaskState state = Protos.TaskState.TASK_LOST;
                String message = String.format("%s while initializing task: %s", t.getClass().getSimpleName(), t.getMessage());
                try {
                    if (task.wasKilled()) {
                        state = Protos.TaskState.TASK_KILLED;
                        message = String.format("Task killed, caught expected %s", t.getClass().getSimpleName());
                    }
                }
                catch (Throwable throwable) {
                    SingularityExecutorMonitor.this.finishTask(task, state, message, (Optional<String>)Optional.of((Object)"Task {} failed before starting process"), task, t);
                    throw throwable;
                }
                SingularityExecutorMonitor.this.finishTask(task, state, message, (Optional<String>)Optional.of((Object)"Task {} failed before starting process"), task, t);
            }
        });
    }

    private void sendStatusUpdate(SingularityExecutorTask task, Protos.TaskState taskState, String message) {
        this.executorUtils.sendStatusUpdate(task.getDriver(), task.getTaskInfo(), taskState, message, task.getLog());
    }

    private void onFinish(SingularityExecutorTask task, Protos.TaskState taskState) {
        this.tasks.remove(task.getTaskId());
        this.processRunningTasks.remove(task.getTaskId());
        this.processBuildingTasks.remove(task.getTaskId());
        task.cleanup(taskState);
        this.logging.stopTaskLogger(task.getTaskId(), task.getLogbackLog());
        this.checkIdleExecutorShutdown(task.getDriver());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void checkIdleExecutorShutdown(ExecutorDriver driver) {
        this.exitLock.lock();
        try {
            this.clearExitCheckerUnsafe();
            if (this.tasks.isEmpty()) {
                this.exitCheckerFuture = Optional.of((Object)this.startExitChecker((Optional<ExecutorDriver>)Optional.of((Object)driver)));
            }
        }
        finally {
            this.exitLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public KillState requestKill(String taskId) {
        Optional maybeTask = Optional.fromNullable((Object)this.tasks.get(taskId));
        if (!maybeTask.isPresent()) {
            return KillState.DIDNT_EXIST;
        }
        SingularityExecutorTask task = (SingularityExecutorTask)maybeTask.get();
        ListenableFuture<ProcessBuilder> processBuilderFuture = null;
        SingularityExecutorTaskProcessCallable runningProcess = null;
        task.getLock().lock();
        boolean wasKilled = task.wasKilled();
        try {
            if (!wasKilled) {
                task.markKilled();
            }
            processBuilderFuture = this.processBuildingTasks.get(task.getTaskId());
            runningProcess = this.processRunningTasks.get(task.getTaskId());
        }
        finally {
            task.getLock().unlock();
        }
        if (processBuilderFuture != null) {
            processBuilderFuture.cancel(true);
            task.getProcessBuilder().cancel();
            return KillState.INTERRUPTING_PRE_PROCESS;
        }
        if (runningProcess != null) {
            if (wasKilled) {
                task.markForceDestroyed();
                runningProcess.signalKillToProcessIfActive();
                return KillState.DESTROYING_PROCESS;
            }
            this.processKiller.submitKillRequest(runningProcess);
            return KillState.KILLING_PROCESS;
        }
        return KillState.INCONSISTENT_STATE;
    }

    private SingularityExecutorTaskProcessCallable buildProcessCallable(SingularityExecutorTask task, ProcessBuilder processBuilder) {
        return new SingularityExecutorTaskProcessCallable(task, processBuilder, this.executorUtils);
    }

    private SingularityExecutorTaskProcessCallable submitProcessMonitor(SingularityExecutorTask task, ProcessBuilder processBuilder) {
        SingularityExecutorTaskProcessCallable processCallable = this.buildProcessCallable(task, processBuilder);
        ListenableFuture processExitFuture = this.runningProcessPool.submit((Callable)processCallable);
        this.watchProcessExitFuture(task, (ListenableFuture<Integer>)processExitFuture);
        return processCallable;
    }

    private void watchProcessExitFuture(final SingularityExecutorTask task, ListenableFuture<Integer> processExitFuture) {
        Futures.addCallback(processExitFuture, (FutureCallback)new FutureCallback<Integer>(){

            public void onSuccess(Integer exitCode) {
                Protos.TaskState taskState = null;
                String message = null;
                if (task.wasKilledDueToThreads()) {
                    taskState = Protos.TaskState.TASK_FAILED;
                    message = String.format("Task used %s threads and was killed (max %s)", task.getThreadCountAtOverageTime(), task.getExecutorData().getMaxTaskThreads().get());
                } else if (task.wasKilled()) {
                    taskState = Protos.TaskState.TASK_KILLED;
                    if (task.wasDestroyedAfterWaiting()) {
                        long millisWaited = (Long)task.getExecutorData().getSigKillProcessesAfterMillis().or((Object)SingularityExecutorMonitor.this.configuration.getHardKillAfterMillis());
                        message = String.format("Task killed forcibly after waiting at least %s", JavaUtils.durationFromMillis((long)millisWaited));
                    } else {
                        message = task.wasForceDestroyed() ? "Task killed forcibly after multiple kill requests from framework" : "Task killed. Process exited gracefully with code " + exitCode;
                    }
                } else if (task.isSuccessExitCode(exitCode)) {
                    taskState = Protos.TaskState.TASK_FINISHED;
                    message = "Process exited normally with code " + exitCode;
                } else {
                    taskState = Protos.TaskState.TASK_FAILED;
                    message = "Process failed with code " + exitCode;
                }
                SingularityExecutorMonitor.this.sendStatusUpdate(task, taskState, message);
                SingularityExecutorMonitor.this.onFinish(task, taskState);
            }

            public void onFailure(Throwable t) {
                task.getLog().error("Task {} failed while running process", (Object)task, (Object)t);
                Protos.TaskState taskState = null;
                String message = null;
                if (task.wasKilled()) {
                    taskState = Protos.TaskState.TASK_KILLED;
                    message = String.format("Task killed, caught %s", t.getClass().getSimpleName());
                } else {
                    taskState = Protos.TaskState.TASK_LOST;
                    message = String.format("%s while running process %s", t.getClass().getSimpleName(), t.getMessage());
                }
                SingularityExecutorMonitor.this.sendStatusUpdate(task, taskState, message);
                SingularityExecutorMonitor.this.onFinish(task, taskState);
            }
        });
    }

    public static enum KillState {
        DIDNT_EXIST,
        INTERRUPTING_PRE_PROCESS,
        KILLING_PROCESS,
        DESTROYING_PROCESS,
        INCONSISTENT_STATE;

    }

    public static enum SubmitState {
        SUBMITTED,
        REJECTED,
        TASK_ALREADY_EXISTED;

    }

    public static enum RunState {
        RUNNING,
        SHUTDOWN;

    }
}

