/*
 * Decompiled with CFR 0.152.
 */
package com.hubspot.singularity.executor;

import com.google.common.base.Charsets;
import com.google.common.base.Optional;
import com.google.common.base.Throwables;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import com.hubspot.mesos.JavaUtils;
import com.hubspot.singularity.executor.SingularityExecutorMonitor;
import com.hubspot.singularity.executor.config.SingularityExecutorConfiguration;
import com.hubspot.singularity.executor.task.SingularityExecutorTaskProcessCallable;
import com.hubspot.singularity.executor.utils.DockerUtils;
import com.hubspot.singularity.runner.base.shared.ProcessFailedException;
import com.spotify.docker.client.DockerException;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class SingularityExecutorThreadChecker {
    private static final Logger LOG = LoggerFactory.getLogger(SingularityExecutorThreadChecker.class);
    private static Pattern CGROUP_CPU_REGEX = Pattern.compile("^\\d+:cpu:/(.*)$");
    private final SingularityExecutorConfiguration configuration;
    private final ScheduledExecutorService scheduledExecutorService;
    private final DockerUtils dockerUtils;
    private SingularityExecutorMonitor monitor;

    @Inject
    public SingularityExecutorThreadChecker(SingularityExecutorConfiguration configuration, DockerUtils dockerUtils) {
        this.configuration = configuration;
        this.dockerUtils = dockerUtils;
        this.scheduledExecutorService = Executors.newScheduledThreadPool(configuration.getThreadCheckThreads(), new ThreadFactoryBuilder().setNameFormat("SingularityExecutorThreadCheckerThread-%d").build());
    }

    public void start(SingularityExecutorMonitor monitor) {
        LOG.info("Starting a thread checker that will run every {}", (Object)JavaUtils.durationFromMillis(this.configuration.getCheckThreadsEveryMillis()));
        this.monitor = monitor;
        this.scheduledExecutorService.scheduleAtFixedRate(new Runnable(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void run() {
                long start = System.currentTimeMillis();
                try {
                    SingularityExecutorThreadChecker.this.checkThreads();
                }
                catch (Throwable t) {
                    LOG.error("While checking threads", t);
                }
                finally {
                    LOG.trace("Finished checking threads after {}", (Object)JavaUtils.duration(start));
                }
            }
        }, this.configuration.getCheckThreadsEveryMillis(), this.configuration.getCheckThreadsEveryMillis(), TimeUnit.MILLISECONDS);
    }

    private void checkThreads() {
        for (SingularityExecutorTaskProcessCallable taskProcess : this.monitor.getRunningTasks()) {
            if (!taskProcess.getTask().getExecutorData().getMaxTaskThreads().isPresent()) continue;
            int maxThreads = taskProcess.getTask().getExecutorData().getMaxTaskThreads().get();
            int usedThreads = 0;
            try {
                usedThreads = this.getNumUsedThreads(taskProcess);
                LOG.trace("{} is using {} threads", (Object)taskProcess.getTask().getTaskId(), (Object)usedThreads);
            }
            catch (InterruptedException ie) {
                Thread.currentThread().interrupt();
                return;
            }
            catch (Throwable t) {
                if (taskProcess.wasKilled()) continue;
                taskProcess.getTask().getLog().error("While fetching used threads for {}", (Object)taskProcess.getTask().getTaskId(), (Object)t);
                continue;
            }
            if (usedThreads <= maxThreads) continue;
            taskProcess.getTask().getLog().info("{} using too many threads: {} (max {})", taskProcess.getTask().getTaskId(), usedThreads, maxThreads);
            taskProcess.getTask().markKilledDueToThreads(usedThreads);
            SingularityExecutorMonitor.KillState killState = this.monitor.requestKill(taskProcess.getTask().getTaskId());
            taskProcess.getTask().getLog().info("Killing {} due to thread overage (kill state {})", (Object)taskProcess.getTask().getTaskId(), (Object)killState);
        }
    }

    public ExecutorService getExecutorService() {
        return this.scheduledExecutorService;
    }

    private int getNumUsedThreads(SingularityExecutorTaskProcessCallable taskProcess) throws InterruptedException, ProcessFailedException {
        Optional<Integer> dockerPid = Optional.absent();
        if (taskProcess.getTask().getTaskInfo().hasContainer() && taskProcess.getTask().getTaskInfo().getContainer().hasDocker()) {
            try {
                String containerName = String.format("%s%s", this.configuration.getDockerPrefix(), taskProcess.getTask().getTaskId());
                int possiblePid = this.dockerUtils.getPid(containerName);
                if (possiblePid == 0) {
                    LOG.warn(String.format("Container %s has pid %s (running: %s). Defaulting to 0 threads running.", containerName, possiblePid, this.dockerUtils.isContainerRunning(containerName)));
                    return 0;
                }
                dockerPid = Optional.of(possiblePid);
            }
            catch (DockerException e) {
                throw new ProcessFailedException(String.format("Could not get docker root pid due to error: %s", e));
            }
        }
        try {
            Path procCgroupPath = Paths.get(String.format(this.configuration.getProcCgroupFormat(), dockerPid.or(taskProcess.getCurrentPid().get())), new String[0]);
            if (Files.exists(procCgroupPath, new LinkOption[0])) {
                for (String line : Files.readAllLines(procCgroupPath, Charsets.UTF_8)) {
                    Matcher matcher = CGROUP_CPU_REGEX.matcher(line);
                    if (!matcher.matches()) continue;
                    return Files.readAllLines(Paths.get(String.format(this.configuration.getCgroupsMesosCpuTasksFormat(), matcher.group(1)), new String[0]), Charsets.UTF_8).size();
                }
                throw new RuntimeException("Unable to parse cgroup container from " + procCgroupPath.toString());
            }
            throw new RuntimeException(procCgroupPath.toString() + " does not exist");
        }
        catch (IOException e) {
            throw Throwables.propagate(e);
        }
    }
}

