/*
 * Decompiled with CFR 0.152.
 */
package com.hubspot.singularity.s3.base;

import com.amazonaws.ClientConfiguration;
import com.amazonaws.auth.BasicAWSCredentials;
import com.amazonaws.services.s3.AmazonS3Client;
import com.amazonaws.services.s3.model.S3Object;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.hubspot.deploy.S3Artifact;
import com.hubspot.mesos.JavaUtils;
import com.hubspot.singularity.runner.base.sentry.SingularityRunnerExceptionNotifier;
import com.hubspot.singularity.s3.base.S3ArtifactChunkDownloader;
import com.hubspot.singularity.s3.base.config.SingularityS3Configuration;
import java.nio.channels.FileChannel;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.slf4j.Logger;

public class S3ArtifactDownloader {
    private final Logger log;
    private final SingularityS3Configuration configuration;
    private final SingularityRunnerExceptionNotifier exceptionNotifier;

    public S3ArtifactDownloader(SingularityS3Configuration configuration, Logger log, SingularityRunnerExceptionNotifier exceptionNotifier) {
        this.configuration = configuration;
        this.log = log;
        this.exceptionNotifier = exceptionNotifier;
    }

    public void download(S3Artifact s3Artifact, Path downloadTo) {
        long start = System.currentTimeMillis();
        boolean success = false;
        try {
            this.downloadThrows(s3Artifact, downloadTo);
            success = true;
        }
        catch (Throwable t) {
            try {
                throw Throwables.propagate(t);
            }
            catch (Throwable throwable) {
                this.log.info("S3 Download {}/{} finished {} after {}", s3Artifact.getS3Bucket(), s3Artifact.getS3ObjectKey(), success ? "successfully" : "with error", JavaUtils.duration(start));
                throw throwable;
            }
        }
        this.log.info("S3 Download {}/{} finished {} after {}", s3Artifact.getS3Bucket(), s3Artifact.getS3ObjectKey(), success ? "successfully" : "with error", JavaUtils.duration(start));
    }

    private BasicAWSCredentials getCredentialsForBucket(String bucketName) {
        if (this.configuration.getS3BucketCredentials().containsKey(bucketName)) {
            return this.configuration.getS3BucketCredentials().get(bucketName).toAWSCredentials();
        }
        return new BasicAWSCredentials(this.configuration.getS3AccessKey().get(), this.configuration.getS3SecretKey().get());
    }

    private void downloadThrows(S3Artifact s3Artifact, Path downloadTo) throws Exception {
        this.log.info("Downloading {}", (Object)s3Artifact);
        ClientConfiguration clientConfiguration = new ClientConfiguration().withSocketTimeout(this.configuration.getS3ChunkDownloadTimeoutMillis());
        if (this.configuration.isS3UseV2Signing()) {
            clientConfiguration.setSignerOverride("S3SignerType");
        }
        AmazonS3Client s3Client = new AmazonS3Client(this.getCredentialsForBucket(s3Artifact.getS3Bucket()), clientConfiguration);
        if (this.configuration.getS3Endpoint().isPresent()) {
            s3Client.setEndpoint(this.configuration.getS3Endpoint().get());
        }
        long length = 0L;
        if (s3Artifact.getFilesize().isPresent()) {
            length = s3Artifact.getFilesize().get();
        } else {
            S3Object details = s3Client.getObject(s3Artifact.getS3Bucket(), s3Artifact.getS3ObjectKey());
            Preconditions.checkNotNull(details, "Couldn't find object at %s/%s", (Object)s3Artifact.getS3Bucket(), (Object)s3Artifact.getS3ObjectKey());
            length = details.getObjectMetadata().getContentLength();
        }
        int numChunks = (int)(length / this.configuration.getS3ChunkSize());
        if (length % this.configuration.getS3ChunkSize() > 0L) {
            ++numChunks;
        }
        long chunkSize = length / (long)numChunks + length % (long)numChunks;
        this.log.info("Downloading {}/{} in {} chunks of {} bytes to {}", s3Artifact.getS3Bucket(), s3Artifact.getS3ObjectKey(), numChunks, chunkSize, downloadTo);
        ExecutorService chunkExecutorService = Executors.newFixedThreadPool(numChunks, new ThreadFactoryBuilder().setDaemon(true).setNameFormat("S3ArtifactDownloaderChunkThread-%d").build());
        ArrayList<Future<Path>> futures = Lists.newArrayListWithCapacity(numChunks);
        for (int chunk = 0; chunk < numChunks; ++chunk) {
            futures.add(chunkExecutorService.submit(new S3ArtifactChunkDownloader(this.configuration, this.log, s3Client, s3Artifact, downloadTo, chunk, chunkSize, length, this.exceptionNotifier)));
        }
        long remainingMillis = this.configuration.getS3DownloadTimeoutMillis();
        boolean failed = false;
        for (int chunk = 0; chunk < numChunks; ++chunk) {
            Future future = (Future)futures.get(chunk);
            if (failed) {
                future.cancel(true);
                continue;
            }
            long start = System.currentTimeMillis();
            if (!this.handleChunk(s3Artifact, future, downloadTo, chunk, start, remainingMillis)) {
                failed = true;
            }
            remainingMillis -= System.currentTimeMillis() - start;
        }
        chunkExecutorService.shutdownNow();
        Preconditions.checkState(!failed, "Downloading %s/%s failed", (Object)s3Artifact.getS3Bucket(), (Object)s3Artifact.getS3ObjectKey());
    }

    private boolean handleChunk(S3Artifact s3Artifact, Future<Path> future, Path downloadTo, int chunk, long start, long remainingMillis) {
        if (remainingMillis <= 0L) {
            remainingMillis = 1L;
        }
        try {
            Path path = future.get(remainingMillis, TimeUnit.MILLISECONDS);
            if (chunk > 0) {
                this.combineChunk(downloadTo, path);
            }
            return true;
        }
        catch (TimeoutException te) {
            this.log.error("Chunk {} for {} timed out after {} - had {} remaining", chunk, s3Artifact.getFilename(), JavaUtils.duration(start), JavaUtils.durationFromMillis(remainingMillis));
            future.cancel(true);
            this.exceptionNotifier.notify("TimeoutException during download", te, ImmutableMap.of("filename", s3Artifact.getFilename(), "chunk", Integer.toString(chunk)));
        }
        catch (Throwable t) {
            this.log.error("Error while handling chunk {} for {}", chunk, s3Artifact.getFilename(), t);
            this.exceptionNotifier.notify(String.format("Error handling chunk (%s)", t.getMessage()), t, ImmutableMap.of("filename", s3Artifact.getFilename(), "chunk", Integer.toString(chunk)));
        }
        return false;
    }

    private void combineChunk(Path downloadTo, Path path) throws Exception {
        long start = System.currentTimeMillis();
        long bytes = 0L;
        this.log.info("Writing {} to {}", (Object)path, (Object)downloadTo);
        try (SeekableByteChannel wbs = Files.newByteChannel(downloadTo, EnumSet.of(StandardOpenOption.APPEND, StandardOpenOption.WRITE), new FileAttribute[0]);
             FileChannel readChannel = FileChannel.open(path, EnumSet.of(StandardOpenOption.READ, StandardOpenOption.DELETE_ON_CLOSE), new FileAttribute[0]);){
            bytes = readChannel.size();
            readChannel.transferTo(0L, bytes, wbs);
        }
        this.log.info("Finished writing {} bytes in {}", (Object)bytes, (Object)JavaUtils.duration(start));
    }
}

