/*
 * Decompiled with CFR 0.152.
 */
package com.hubspot.singularity.s3.base;

import com.google.common.base.Optional;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableCollection;
import com.google.common.collect.ImmutableList;
import com.google.common.hash.HashCode;
import com.google.common.hash.Hashing;
import com.hubspot.deploy.Artifact;
import com.hubspot.deploy.EmbeddedArtifact;
import com.hubspot.deploy.ExternalArtifact;
import com.hubspot.deploy.RemoteArtifact;
import com.hubspot.deploy.S3Artifact;
import com.hubspot.singularity.runner.base.configuration.SingularityRunnerBaseConfiguration;
import com.hubspot.singularity.runner.base.sentry.SingularityRunnerExceptionNotifier;
import com.hubspot.singularity.runner.base.shared.ProcessFailedException;
import com.hubspot.singularity.runner.base.shared.SimpleProcessManager;
import com.hubspot.singularity.s3.base.CacheCheck;
import com.hubspot.singularity.s3.base.CacheCheckResult;
import com.hubspot.singularity.s3.base.S3ArtifactDownloader;
import com.hubspot.singularity.s3.base.config.SingularityS3Configuration;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.CopyOption;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.EnumSet;
import java.util.List;
import org.slf4j.Logger;

public class ArtifactManager
extends SimpleProcessManager {
    private final Path cacheDirectory;
    private final Logger log;
    private final S3ArtifactDownloader s3ArtifactDownloader;
    private final Optional<String> useCompressProgram;

    public ArtifactManager(SingularityRunnerBaseConfiguration runnerBaseConfiguration, SingularityS3Configuration configuration, Logger log, SingularityRunnerExceptionNotifier exceptionNotifier) {
        super(log);
        this.cacheDirectory = Paths.get(configuration.getArtifactCacheDirectory(), new String[0]);
        this.log = log;
        this.s3ArtifactDownloader = new S3ArtifactDownloader(configuration, log, exceptionNotifier);
        this.useCompressProgram = runnerBaseConfiguration.getUseCompressProgram();
    }

    private long getSize(Path path) {
        try {
            return Files.size(path);
        }
        catch (IOException ioe) {
            throw new RuntimeException(String.format("Couldnt get file size of %s", path), ioe);
        }
    }

    private boolean filesSizeMatches(RemoteArtifact artifact, Path path) {
        return !artifact.getFilesize().isPresent() || artifact.getFilesize().get().longValue() == this.getSize(path);
    }

    private boolean md5Matches(Artifact artifact, Path path) {
        return !artifact.getMd5sum().isPresent() || artifact.getMd5sum().get().equalsIgnoreCase(this.calculateMd5sum(path));
    }

    private void checkFilesize(RemoteArtifact artifact, Path path) {
        if (!this.filesSizeMatches(artifact, path)) {
            throw new RuntimeException(String.format("Filesize %s (%s) does not match expected (%s)", this.getSize(path), path, artifact.getFilesize()));
        }
    }

    private void checkMd5(Artifact artifact, Path path) {
        if (!this.md5Matches(artifact, path)) {
            throw new RuntimeException(String.format("Md5sum %s (%s) does not match expected (%s)", this.calculateMd5sum(path), path, artifact.getMd5sum().get()));
        }
    }

    private Path createTempPath(String filename) {
        try {
            return Files.createTempFile(this.cacheDirectory, filename, null, new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new RuntimeException(String.format("Couldn't create temporary file for %s", filename), e);
        }
    }

    private void downloadAndCheck(RemoteArtifact artifact, Path downloadTo) {
        if (artifact instanceof ExternalArtifact) {
            this.downloadExternalArtifact((ExternalArtifact)artifact, downloadTo);
        } else if (artifact instanceof S3Artifact) {
            this.downloadS3Artifact((S3Artifact)artifact, downloadTo);
        } else {
            throw new IllegalArgumentException("Unknown artifact type: " + artifact.getClass());
        }
        this.checkFilesize(artifact, downloadTo);
        this.checkMd5(artifact, downloadTo);
    }

    public void extract(EmbeddedArtifact embeddedArtifact, Path directory) {
        Path extractTo = directory.resolve(embeddedArtifact.getFilename());
        Path parent = extractTo.getParent();
        try {
            if (parent != null) {
                Files.createDirectories(parent, new FileAttribute[0]);
            }
        }
        catch (IOException e) {
            throw new RuntimeException(String.format("Couldn't extract %s, unable to create directory %s", embeddedArtifact.getName(), parent), e);
        }
        this.log.info("Extracting {} bytes of {} to {}", embeddedArtifact.getContent().length, embeddedArtifact.getName(), extractTo);
        try (SeekableByteChannel byteChannel = Files.newByteChannel(extractTo, EnumSet.of(StandardOpenOption.CREATE_NEW, StandardOpenOption.WRITE), new FileAttribute[0]);){
            byteChannel.write(ByteBuffer.wrap(embeddedArtifact.getContent()));
        }
        catch (IOException e) {
            throw new RuntimeException(String.format("Couldn't extract %s", embeddedArtifact.getName()), e);
        }
        this.checkMd5(embeddedArtifact, extractTo);
    }

    private Path downloadAndCache(RemoteArtifact artifact, String filename, String cacheMissMessage) {
        Path tempFilePath = this.createTempPath(filename);
        this.downloadAndCheck(artifact, tempFilePath);
        Path cachedPath = this.getCachedPath(filename);
        try {
            Files.move(tempFilePath, cachedPath, StandardCopyOption.ATOMIC_MOVE);
        }
        catch (IOException e) {
            throw new RuntimeException(String.format("Couldn't move %s to cache at %s (Not cached because %s)", tempFilePath, cachedPath, cacheMissMessage), e);
        }
        return cachedPath;
    }

    private Path getCachedPath(String filename) {
        return this.cacheDirectory.resolve(filename);
    }

    private CacheCheck checkCached(RemoteArtifact artifact, Path cachedPath) {
        if (!Files.exists(cachedPath, new LinkOption[0])) {
            String message = String.format("Cached %s did not exist", cachedPath);
            this.log.debug(message);
            return new CacheCheck(CacheCheckResult.DOES_NOT_EXIST, message);
        }
        if (!this.filesSizeMatches(artifact, cachedPath)) {
            String message = String.format("Cached %s (%s) did not match file size %s", cachedPath, this.getSize(cachedPath), artifact.getFilesize());
            this.log.debug(message);
            return new CacheCheck(CacheCheckResult.FILE_SIZE_MISMATCH, message);
        }
        if (!this.md5Matches(artifact, cachedPath)) {
            String message = String.format("Cached %s (%s) did not match md5 %s", cachedPath, this.calculateMd5sum(cachedPath), artifact.getMd5sum().get());
            this.log.debug(message);
            return new CacheCheck(CacheCheckResult.MD5_MISMATCH, message);
        }
        return new CacheCheck(CacheCheckResult.FOUND, "");
    }

    public Path fetch(RemoteArtifact artifact) {
        String filename = artifact.getFilenameForCache();
        Path cachedPath = this.getCachedPath(filename);
        CacheCheck cacheCheck = this.checkCached(artifact, cachedPath);
        if (cacheCheck.getCacheCheckResult() != CacheCheckResult.FOUND) {
            this.log.info(cacheCheck.getMessage());
            this.downloadAndCache(artifact, filename, cacheCheck.getMessage());
        } else {
            this.log.info("Using cached file {}", (Object)cachedPath);
        }
        return cachedPath;
    }

    private void downloadExternalArtifact(ExternalArtifact externalArtifact, Path downloadTo) {
        this.downloadUri(externalArtifact.getUrl(), downloadTo);
    }

    private void downloadS3Artifact(S3Artifact s3Artifact, Path downloadTo) {
        this.s3ArtifactDownloader.download(s3Artifact, downloadTo);
    }

    private void downloadUri(String uri, Path path) {
        this.log.info("Downloading {} to {}", (Object)uri, (Object)path);
        ImmutableList<String> command = ImmutableList.of("wget", uri, "-O", path.toString(), "-nv", "--no-check-certificate");
        this.runCommandAndThrowRuntimeException(command);
    }

    public void copy(Path source, Path destination, String destinationFilename) {
        this.log.info("Copying {} to {}", (Object)source, (Object)destination);
        Path destinationPath = destination.resolve(destinationFilename);
        try {
            Files.createDirectories(destination, new FileAttribute[0]);
            Files.copy(source, destinationPath, new CopyOption[0]);
        }
        catch (FileAlreadyExistsException e) {
            if (!this.calculateMd5sum(source).equals(this.calculateMd5sum(destinationPath))) {
                throw new RuntimeException(e);
            }
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public void untar(Path source, Path destination) {
        this.log.info("Untarring {} to {}", (Object)source, (Object)destination);
        ImmutableCollection.Builder commandBuilder = ImmutableList.builder().add(new String[]{"tar", "-oxf", source.toString(), "-C", destination.toString()});
        if (this.useCompressProgram.isPresent()) {
            ((ImmutableList.Builder)commandBuilder).add("--use-compress-program=" + this.useCompressProgram.get());
        } else {
            ((ImmutableList.Builder)commandBuilder).add("-z");
        }
        this.runCommandAndThrowRuntimeException((List<String>)((Object)((ImmutableList.Builder)commandBuilder).build()));
    }

    private void runCommandAndThrowRuntimeException(List<String> command) {
        try {
            super.runCommand(command);
        }
        catch (ProcessFailedException | InterruptedException e) {
            throw Throwables.propagate(e);
        }
    }

    private String calculateMd5sum(Path path) {
        try {
            HashCode hc = com.google.common.io.Files.hash(path.toFile(), Hashing.md5());
            return hc.toString();
        }
        catch (IOException e) {
            throw Throwables.propagate(e);
        }
    }
}

