/*
 * Decompiled with CFR 0.152.
 */
package com.hubspot.singularity.runner.base.shared;

import com.google.common.base.Joiner;
import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.hubspot.mesos.JavaUtils;
import com.hubspot.singularity.runner.base.shared.ProcessUtils;
import com.hubspot.singularity.runner.base.shared.Signal;
import java.io.IOException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.slf4j.Logger;

public abstract class SafeProcessManager {
    private final Logger log;
    private final Lock processLock;
    private volatile Optional<String> currentProcessCmd;
    private volatile Optional<Process> currentProcess;
    private volatile Optional<Integer> currentProcessPid;
    private volatile Optional<Long> currentProcessStart;
    private final AtomicBoolean killed;
    private final ProcessUtils processUtils;

    public SafeProcessManager(Logger log) {
        this.log = log;
        this.processUtils = new ProcessUtils(log);
        this.currentProcessCmd = Optional.absent();
        this.currentProcess = Optional.absent();
        this.currentProcessPid = Optional.absent();
        this.processLock = new ReentrantLock();
        this.killed = new AtomicBoolean(false);
    }

    public Logger getLog() {
        return this.log;
    }

    public boolean wasKilled() {
        return this.killed.get();
    }

    public void markKilled() {
        this.processLock.lock();
        try {
            this.killed.set(true);
        }
        finally {
            this.processLock.unlock();
        }
    }

    private void lockInterruptibly() {
        try {
            this.processLock.lockInterruptibly();
        }
        catch (InterruptedException e) {
            throw Throwables.propagate(e);
        }
    }

    public Process startProcess(ProcessBuilder builder) {
        String cmd = builder.command().get(0);
        this.log.debug("Starting process {}", (Object)Joiner.on(" ").join(builder.command()));
        this.processLock.lock();
        Process process = null;
        try {
            Preconditions.checkState(!this.killed.get(), "Can not start new process, killed is set");
            Preconditions.checkState(!this.currentProcess.isPresent(), "Can not start new process, already had process");
            this.currentProcessStart = Optional.of(System.currentTimeMillis());
            process = builder.start();
            this.currentProcessPid = Optional.of(this.processUtils.getUnixPID(process));
            this.currentProcess = Optional.of(process);
            this.currentProcessCmd = Optional.of(cmd);
            this.log.debug("Started process {}", (Object)this.getCurrentProcessToString());
        }
        catch (IOException e) {
            throw Throwables.propagate(e);
        }
        finally {
            this.processLock.unlock();
        }
        return process;
    }

    private void resetCurrentVariables() {
        this.currentProcess = Optional.absent();
        this.currentProcessPid = Optional.absent();
        this.currentProcessCmd = Optional.absent();
        this.currentProcessStart = Optional.absent();
    }

    protected void processFinished(Optional<Integer> exitCode) {
        this.lockInterruptibly();
        try {
            if (this.currentProcessCmd.isPresent() && this.currentProcessStart.isPresent()) {
                if (exitCode.isPresent()) {
                    this.log.debug("Process {} exited with {} after {}", this.currentProcessCmd.get(), exitCode.get(), JavaUtils.duration(this.currentProcessStart.get()));
                } else {
                    this.log.debug("Process {} abandoned after {}", (Object)this.currentProcessCmd.get(), (Object)JavaUtils.duration(this.currentProcessStart.get()));
                }
            } else {
                this.log.warn("Process finished called on an empty process manager");
            }
            this.resetCurrentVariables();
        }
        finally {
            this.processLock.unlock();
        }
    }

    public Optional<Integer> getCurrentPid() {
        return this.currentProcessPid;
    }

    public String getCurrentProcessToString() {
        return String.format("%s - (pid: %s)", this.currentProcessCmd.or("<none>"), this.currentProcessPid.or(0));
    }

    public void signalTermToProcessIfActive() {
        this.processLock.lock();
        try {
            if (this.currentProcessPid.isPresent()) {
                this.processUtils.sendSignal(Signal.SIGTERM, this.currentProcessPid.get());
            }
        }
        finally {
            this.processLock.unlock();
        }
    }

    public void signalKillToProcessIfActive() {
        this.processLock.lock();
        try {
            if (this.currentProcess.isPresent()) {
                this.processUtils.sendSignal(Signal.SIGKILL, this.currentProcessPid.get());
            }
        }
        finally {
            this.processLock.unlock();
        }
    }
}

