/*
 * Decompiled with CFR 0.152.
 */
package com.hubspot.mesos;

import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.common.net.InetAddresses;
import com.google.common.primitives.Longs;
import com.hubspot.mesos.Resources;
import java.net.UnknownHostException;
import java.nio.ByteBuffer;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Random;
import java.util.Set;
import org.apache.mesos.Protos;

public final class MesosUtils {
    public static final String CPUS = "cpus";
    public static final String MEMORY = "mem";
    public static final String PORTS = "ports";
    public static final String DISK = "disk";
    private static final Comparator<Protos.Value.Range> RANGE_COMPARATOR = new Comparator<Protos.Value.Range>(){

        @Override
        public int compare(Protos.Value.Range o1, Protos.Value.Range o2) {
            return Longs.compare(o1.getBegin(), o2.getBegin());
        }
    };

    private MesosUtils() {
    }

    private static double getScalar(Protos.Resource r) {
        return r.getScalar().getValue();
    }

    private static double getScalar(List<Protos.Resource> resources, String name, Optional<String> requiredRole) {
        for (Protos.Resource r : resources) {
            if (!r.hasName() || !r.getName().equals(name) || !r.hasScalar() || !MesosUtils.hasRequiredRole(r, requiredRole)) continue;
            return MesosUtils.getScalar(r);
        }
        return 0.0;
    }

    private static boolean hasRole(Protos.Resource r) {
        return r.hasRole() && !r.getRole().equals("*");
    }

    private static boolean hasRequiredRole(Protos.Resource r, Optional<String> requiredRole) {
        if (requiredRole.isPresent() && MesosUtils.hasRole(r)) {
            return requiredRole.get().equals(r.getRole());
        }
        if (requiredRole.isPresent() && !MesosUtils.hasRole(r)) {
            return false;
        }
        if (!requiredRole.isPresent() && MesosUtils.hasRole(r)) {
            return false;
        }
        return !requiredRole.isPresent() && !MesosUtils.hasRole(r);
    }

    private static Protos.Value.Ranges getRanges(List<Protos.Resource> resources, String name) {
        for (Protos.Resource r : resources) {
            if (!r.hasName() || !r.getName().equals(name) || !r.hasRanges()) continue;
            return r.getRanges();
        }
        return Protos.Value.Ranges.getDefaultInstance();
    }

    private static int getNumRanges(List<Protos.Resource> resources, String name) {
        int totalRanges = 0;
        for (Protos.Value.Range range : MesosUtils.getRanges(resources, name).getRangeList()) {
            totalRanges = (int)((long)totalRanges + (range.getEnd() - range.getBegin() + 1L));
        }
        return totalRanges;
    }

    public static Protos.Resource getCpuResource(double cpus, Optional<String> role) {
        return MesosUtils.newScalar(CPUS, cpus, role);
    }

    public static Protos.Resource getMemoryResource(double memory, Optional<String> role) {
        return MesosUtils.newScalar(MEMORY, memory, role);
    }

    public static Protos.Resource getDiskResource(double disk, Optional<String> role) {
        return MesosUtils.newScalar(DISK, disk, role);
    }

    public static long[] getPorts(Protos.Resource portsResource, int numPorts) {
        long[] ports = new long[numPorts];
        if (numPorts == 0) {
            return ports;
        }
        int idx = 0;
        for (Protos.Value.Range r : portsResource.getRanges().getRangeList()) {
            long port = r.getBegin();
            while (port <= r.getEnd()) {
                ports[idx++] = port++;
                if (idx < numPorts) continue;
                return ports;
            }
        }
        return ports;
    }

    public static Protos.Resource getPortRangeResource(long begin, long end) {
        return MesosUtils.newRange(PORTS, begin, end);
    }

    public static List<Long> getAllPorts(List<Protos.Resource> resources) {
        Protos.Value.Ranges ranges = MesosUtils.getRanges(resources, PORTS);
        ArrayList<Long> ports = Lists.newArrayList();
        if (ranges != null) {
            for (Protos.Value.Range range : ranges.getRangeList()) {
                for (long port = range.getBegin(); port <= range.getEnd(); ++port) {
                    ports.add(port);
                }
            }
        }
        return ports;
    }

    public static Protos.Resource getPortsResource(int numPorts, Protos.Offer offer) {
        return MesosUtils.getPortsResource(numPorts, offer.getResourcesList(), Collections.emptyList());
    }

    public static Protos.Resource getPortsResource(int numPorts, List<Protos.Resource> resources, List<Long> otherRequestedPorts) {
        ArrayList<Long> requestedPorts = new ArrayList<Long>(otherRequestedPorts);
        Protos.Value.Ranges ranges = MesosUtils.getRanges(resources, PORTS);
        Preconditions.checkState(ranges.getRangeCount() > 0, "Ports %s should have existed in resources %s", PORTS, MesosUtils.formatForLogging(resources));
        Protos.Value.Ranges.Builder rangesBldr = Protos.Value.Ranges.newBuilder();
        int portsSoFar = 0;
        ArrayList<Protos.Value.Range> offerRangeList = Lists.newArrayList(ranges.getRangeList());
        Random random = new Random();
        Collections.shuffle(offerRangeList, random);
        if (numPorts > 0) {
            for (Protos.Value.Range range : offerRangeList) {
                long rangeStartSelection = Math.max(range.getBegin(), range.getEnd() - (long)(numPorts - portsSoFar + 1));
                if (rangeStartSelection != range.getBegin()) {
                    int rangeDelta = (int)(rangeStartSelection - range.getBegin()) + 1;
                    rangeStartSelection = (long)random.nextInt(rangeDelta) + range.getBegin();
                }
                long rangeEndSelection = Math.min(range.getEnd(), rangeStartSelection + (long)(numPorts - portsSoFar - 1));
                rangesBldr.addRange(Protos.Value.Range.newBuilder().setBegin(rangeStartSelection).setEnd(rangeEndSelection));
                portsSoFar = (int)((long)portsSoFar + (rangeEndSelection - rangeStartSelection + 1L));
                ArrayList<Long> toRemove = new ArrayList<Long>();
                Iterator iterator = requestedPorts.iterator();
                while (iterator.hasNext()) {
                    long port = (Long)iterator.next();
                    if (rangeStartSelection < port || rangeEndSelection > port) continue;
                    toRemove.add(port);
                    --portsSoFar;
                }
                requestedPorts.removeAll(toRemove);
                if (portsSoFar != numPorts) continue;
                break;
            }
        }
        Iterator iterator = requestedPorts.iterator();
        while (iterator.hasNext()) {
            long port = (Long)iterator.next();
            rangesBldr.addRange(Protos.Value.Range.newBuilder().setBegin(port).setEnd(port).build());
        }
        return Protos.Resource.newBuilder().setType(Protos.Value.Type.RANGES).setName(PORTS).setRanges(rangesBldr).build();
    }

    private static Protos.Resource newScalar(String name, double value, Optional<String> role) {
        Protos.Resource.Builder builder = Protos.Resource.newBuilder().setName(name).setType(Protos.Value.Type.SCALAR).setScalar(Protos.Value.Scalar.newBuilder().setValue(value).build());
        if (role.isPresent()) {
            builder.setRole(role.get());
        }
        return builder.build();
    }

    private static Protos.Resource newRange(String name, long begin, long end) {
        return Protos.Resource.newBuilder().setName(name).setType(Protos.Value.Type.RANGES).setRanges(Protos.Value.Ranges.newBuilder().addRange(Protos.Value.Range.newBuilder().setBegin(begin).setEnd(end).build()).build()).build();
    }

    public static Set<String> getRoles(Protos.Offer offer) {
        HashSet<String> roles = Sets.newHashSet();
        for (Protos.Resource r : offer.getResourcesList()) {
            roles.add(r.getRole());
        }
        return roles;
    }

    public static double getNumCpus(Protos.Offer offer) {
        return MesosUtils.getNumCpus(offer.getResourcesList(), Optional.absent());
    }

    public static double getMemory(Protos.Offer offer) {
        return MesosUtils.getMemory(offer.getResourcesList(), Optional.absent());
    }

    public static double getDisk(Protos.Offer offer) {
        return MesosUtils.getDisk(offer.getResourcesList(), Optional.absent());
    }

    public static double getNumCpus(List<Protos.Resource> resources, Optional<String> requiredRole) {
        return MesosUtils.getScalar(resources, CPUS, requiredRole);
    }

    public static double getMemory(List<Protos.Resource> resources, Optional<String> requiredRole) {
        return MesosUtils.getScalar(resources, MEMORY, requiredRole);
    }

    public static double getDisk(List<Protos.Resource> resources, Optional<String> requiredRole) {
        return MesosUtils.getScalar(resources, DISK, requiredRole);
    }

    public static int getNumPorts(List<Protos.Resource> resources) {
        return MesosUtils.getNumRanges(resources, PORTS);
    }

    public static int getNumPorts(Protos.Offer offer) {
        return MesosUtils.getNumPorts(offer.getResourcesList());
    }

    public static boolean doesOfferMatchResources(Optional<String> requiredRole, Resources resources, List<Protos.Resource> offerResources, List<Long> otherRequestedPorts) {
        double numCpus = MesosUtils.getNumCpus(offerResources, requiredRole);
        if (numCpus < resources.getCpus()) {
            return false;
        }
        double memory = MesosUtils.getMemory(offerResources, requiredRole);
        if (memory < resources.getMemoryMb()) {
            return false;
        }
        double disk = MesosUtils.getDisk(offerResources, requiredRole);
        if (disk < resources.getDiskMb()) {
            return false;
        }
        int numPorts = MesosUtils.getNumPorts(offerResources);
        if (numPorts < resources.getNumPorts()) {
            return false;
        }
        return MesosUtils.getAllPorts(offerResources).containsAll(otherRequestedPorts);
    }

    public static boolean isTaskDone(Protos.TaskState state) {
        return state == Protos.TaskState.TASK_FAILED || state == Protos.TaskState.TASK_LOST || state == Protos.TaskState.TASK_KILLED || state == Protos.TaskState.TASK_FINISHED;
    }

    public static String getMasterHostAndPort(Protos.MasterInfo masterInfo) {
        byte[] fromIp = ByteBuffer.allocate(4).putInt(masterInfo.getIp()).array();
        try {
            return String.format("%s:%s", InetAddresses.fromLittleEndianByteArray(fromIp).getHostAddress(), masterInfo.getPort());
        }
        catch (UnknownHostException e) {
            throw Throwables.propagate(e);
        }
    }

    private static Optional<Protos.Resource> getMatchingResource(Protos.Resource toMatch, List<Protos.Resource> resources) {
        for (Protos.Resource resource : resources) {
            if (!toMatch.getName().equals(resource.getName())) continue;
            return Optional.of(resource);
        }
        return Optional.absent();
    }

    private static Protos.Value.Ranges subtractRanges(Protos.Value.Ranges ranges, Protos.Value.Ranges toSubtract) {
        Protos.Value.Ranges.Builder newRanges = Protos.Value.Ranges.newBuilder();
        ArrayList<Protos.Value.Range> sortedRanges = Lists.newArrayList(ranges.getRangeList());
        Collections.sort(sortedRanges, RANGE_COMPARATOR);
        ArrayList<Protos.Value.Range> subtractRanges = Lists.newArrayList(toSubtract.getRangeList());
        Collections.sort(subtractRanges, RANGE_COMPARATOR);
        int s = 0;
        for (Protos.Value.Range range : ranges.getRangeList()) {
            Protos.Value.Range.Builder currentRange = range.toBuilder();
            for (int i = s; i < subtractRanges.size(); ++i) {
                Protos.Value.Range matchedRange = (Protos.Value.Range)subtractRanges.get(i);
                if (matchedRange.getBegin() < currentRange.getBegin() || matchedRange.getEnd() > currentRange.getEnd()) {
                    s = i;
                    break;
                }
                currentRange.setEnd(matchedRange.getBegin() - 1L);
                if (currentRange.getEnd() >= currentRange.getBegin()) {
                    newRanges.addRange(currentRange.build());
                }
                currentRange = Protos.Value.Range.newBuilder();
                currentRange.setBegin(matchedRange.getEnd() + 1L);
                currentRange.setEnd(range.getEnd());
            }
            if (currentRange.getEnd() < currentRange.getBegin()) continue;
            newRanges.addRange(currentRange.build());
        }
        return newRanges.build();
    }

    public static List<Protos.Resource> subtractResources(List<Protos.Resource> resources, List<Protos.Resource> subtract) {
        ArrayList<Protos.Resource> remaining = Lists.newArrayListWithCapacity(resources.size());
        for (Protos.Resource resource : resources) {
            Optional<Protos.Resource> matched = MesosUtils.getMatchingResource(resource, subtract);
            if (!matched.isPresent()) {
                remaining.add(resource.toBuilder().clone().build());
                continue;
            }
            Protos.Resource.Builder resourceBuilder = resource.toBuilder().clone();
            if (resource.hasScalar()) {
                resourceBuilder.setScalar(resource.toBuilder().getScalarBuilder().setValue(resource.getScalar().getValue() - matched.get().getScalar().getValue()).build());
            } else if (resource.hasRanges()) {
                resourceBuilder.setRanges(MesosUtils.subtractRanges(resource.getRanges(), matched.get().getRanges()));
            } else {
                throw new IllegalStateException(String.format("Can't subtract non-scalar or range resources %s", MesosUtils.formatForLogging(resource)));
            }
            remaining.add(resourceBuilder.build());
        }
        return remaining;
    }

    public static Resources buildResourcesFromMesosResourceList(List<Protos.Resource> resources) {
        return new Resources(MesosUtils.getNumCpus(resources, Optional.absent()), MesosUtils.getMemory(resources, Optional.absent()), MesosUtils.getNumPorts(resources), MesosUtils.getDisk(resources, Optional.absent()));
    }

    public static Path getTaskDirectoryPath(String taskId) {
        return Paths.get(MesosUtils.getSafeTaskIdForDirectory(taskId), new String[0]).toAbsolutePath();
    }

    public static String getSafeTaskIdForDirectory(String taskId) {
        return taskId.replace(":", "_");
    }

    public static String formatForLogging(Object object) {
        return object.toString().replace("\n", "").replaceAll("( )+", " ");
    }
}

