package code.complete.log.formatter;

import android.support.annotation.Nullable;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import code.complete.log.LogFormatter;

/**
 * 美化输出的Formatter
 * @author imkarl
 */
public final class PrettyFormatter implements LogFormatter {

    private static final String WRAP_START = "╔═══════════════════════════════════════════════════════════════════";
    private static final String WRAP_MIDDLE = "║";
    private static final String WRAP_END = "╚═══════════════════════════════════════════════════════════════════";

    @Override
    public String format(@Nullable Object message) {
        if (message == null) {
            return "[NULL]";
        }
        if (message instanceof String) {
            return getSpecialString((String) message);
        }
        if (message instanceof JSONObject) {
            return getJSONString((JSONObject) message);
        }
        if (message instanceof JSONArray) {
            return getJSONString((JSONArray) message);
        }
        if (message instanceof Throwable) {
            return getThrowableString((Throwable) message);
        }
        if (message instanceof StackTraceElement[]) {
            return getStackTraceString((StackTraceElement[]) message);
        }
        return message.toString();
    }


    private static String getSpecialString(String str) {
        if (str.trim().isEmpty()) {
            str = str.replace("\n", "\\n");
            str = str.replace("\t", "\\t");
            return "["+str+"]";
        }
        return str;
    }

    private static String getJSONString(JSONObject json) {
        try {
            return wrap(json.toString(2));
        } catch (JSONException e) {
            return json.toString();
        }
    }
    private static String getJSONString(JSONArray json) {
        try {
            return wrap(json.toString(2));
        } catch (JSONException e) {
            return json.toString();
        }
    }

    private static String getThrowableString(Throwable throwable) {
        final String throwableName = throwable.getClass().getName();
        final String message = throwable.getLocalizedMessage();

        StringBuilder str = new StringBuilder(WRAP_START).append('\n');
        str.append(WRAP_MIDDLE).append(message != null ? (throwableName + ": " + message) : throwableName).append('\n');
        final StackTraceElement[] trace = throwable.getStackTrace();
        if (trace.length > 0) {
            for (StackTraceElement element : trace) {
                str.append(WRAP_MIDDLE).append("\tat ").append(element).append('\n');
            }
        } else {
            str.append(WRAP_MIDDLE).append("\t(No StackTrace)").append('\n');
        }
        str.append(WRAP_END);
        return str.toString();
    }
    private static String getStackTraceString(StackTraceElement[] trace) {
        StringBuilder str = new StringBuilder(WRAP_START).append('\n');
        if (trace.length > 0) {
            str.append(WRAP_MIDDLE).append("caller method trace:").append('\n');
            for (StackTraceElement element : trace) {
                str.append(WRAP_MIDDLE).append("\tat ").append(element).append('\n');
            }
        } else {
            str.append(WRAP_MIDDLE).append("\t(No StackTrace)").append('\n');
        }
        str.append(WRAP_END);
        return str.toString();
    }


    private static String wrap(String str) {
        StringBuilder strBuilder = new StringBuilder(WRAP_START).append('\n');
        for (String line : str.split("\n")) {
            strBuilder.append(WRAP_MIDDLE).append(" ").append(line).append('\n');
        }
        strBuilder.append(WRAP_END);
        return strBuilder.toString();
    }

}
