package code.complete.log;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import code.complete.log.filter.SimpleFilter;
import code.complete.log.formatter.SimpleFormatter;
import code.complete.log.printer.AndroidPrinter;

/**
 * 日志相关工具类
 * @author imkarl
 */
public final class LogUtils {

    // 全局TAG
    private static String sGlobalTag = "CodeComplete";
    // 日志打印器
    private static LogPrinter sPrinter = new AndroidPrinter(true);
    // 日志过滤器
    private static LogFilter sFilter = new SimpleFilter(true);
    // 日志格式化工具
    private static LogFormatter sFormatter = new SimpleFormatter();

    /**
     * 设置全局TAG
     */
    public static void setGlobalTag(@NonNull String tag) {
        sGlobalTag = tag;
    }
    /**
     * 获取当前的全局TAG
     */
    @NonNull
    public static String getGlobalTag() {
        return sGlobalTag;
    }

    /**
     * 设置打印器
     */
    public static void setPrinter(@NonNull LogPrinter printer) {
        sPrinter = printer;
    }
    /**
     * 设置过滤器
     */
    public static void setFilter(@Nullable LogFilter filter) {
        sFilter = filter;
    }
    /**
     * 设置格式化工具
     */
    public static void setFormatter(@NonNull LogFormatter formatter) {
        sFormatter = formatter;
    }

    public static void v(@Nullable Object message) {
        println(Log.VERBOSE, sGlobalTag, message);
    }
    public static void d(@Nullable Object message) {
        println(Log.DEBUG, sGlobalTag, message);
    }
    public static void i(@Nullable Object message) {
        println(Log.INFO, sGlobalTag, message);
    }
    public static void w(@Nullable Object message) {
        println(Log.WARN, sGlobalTag, message);
    }
    public static void e(@Nullable Object message) {
        println(Log.ERROR, sGlobalTag, message);
    }

    /**
     * 定制化日志打印（日志级别:Log.DEBUG）
     * @param tag 日志TAG
     * @param message 日志内容
     */
    public static void log(@NonNull String tag, @Nullable Object message) {
        println(Log.DEBUG, tag, message);
    }

    /**
     * 定制化日志打印
     * @param priority 日志级别
     * @param tag 日志TAG
     * @param message 日志内容
     */
    public static void log(@LogPriority int priority, @NonNull String tag, @Nullable Object message) {
        println(priority, tag, message);
    }

    /**
     * 打印调用堆栈（日志级别:Log.DEBUG）
     */
    public static void logCallerTrace() {
        println(Log.DEBUG, sGlobalTag, filterCaller(new Throwable().getStackTrace()));
    }


    /**
     * 日志打印的具体实现
     * @param priority 日志级别
     * @param tag 日志TAG
     * @param message 日志内容
     */
    private static void println(@LogPriority int priority, String tag, Object message) {
        final Thread currentThread = Thread.currentThread();
        final StackTraceElement[] callerStackTrace = filterCaller(currentThread.getStackTrace());
        final long time = System.currentTimeMillis();
        final String threadName = currentThread.getName();
        final LogInfo info = LogInfo.obtain(priority, tag, time, threadName, callerStackTrace);

        if (sFilter == null || sFilter.isLoggable(info)) {
            sPrinter.println(info, sFormatter.format(message));
        }

        info.recycle();
    }


    /**
     * 过滤调用栈
     * @return 不包含LogUtils的调用栈
     */
    private static StackTraceElement[] filterCaller(StackTraceElement[] callStackTrace) {
        int startIndex = 0;
        for (int i = 0; i < callStackTrace.length; i++) {
            String className = callStackTrace[i].getClassName();
            if (!className.startsWith(LogUtils.class.getName())
                    && !className.startsWith("java.lang.reflect")
                    && !className.startsWith("com.android.tools")) {
                startIndex = i;
                break;
            }
        }

        StackTraceElement[] stackTraceElement = new StackTraceElement[callStackTrace.length - startIndex];
        System.arraycopy(callStackTrace, startIndex, stackTraceElement, 0, stackTraceElement.length);
        return stackTraceElement;
    }

}
