package cn.zoecloud.core.operation;

import cn.zoecloud.core.ClientException;
import cn.zoecloud.core.ServiceException;
import cn.zoecloud.core.auth.Credential;
import cn.zoecloud.core.auth.DefaultRequestSigner;
import cn.zoecloud.core.auth.RequestSigner;
import cn.zoecloud.core.parser.ResponseParseException;
import cn.zoecloud.core.parser.ResponseParser;
import cn.zoecloud.core.service.RequestMessage;
import cn.zoecloud.core.service.ResponseMessage;
import cn.zoecloud.core.service.ServiceClient;

import static cn.zoecloud.core.util.LogUtil.logException;

/**
 * 请求处理抽象类
 * @author Leo
 */
public abstract class ZoeCloudOperation {
    private static final String DEFAULT_CONTENT_TYPE = "application/json;charset=UTF-8";

    private ServiceClient client;
    private String endpoint;
    private RequestSigner requestSigner;

    public ZoeCloudOperation(ServiceClient client, Credential credential, String endpoint) {
        this.client = client;
        this.endpoint = endpoint;
        this.requestSigner = new DefaultRequestSigner(credential, getContextPath(endpoint));
    }

    public String getEndpoint() {
        return endpoint;
    }

    /**
     * 发送请求
     * @param request
     * @return
     * @throws ServiceException
     * @throws ClientException
     */
    private ResponseMessage send(RequestMessage request) throws ServiceException, ClientException {
        ResponseMessage response = null;
        response = client.sendRequest(request);
        return response;
    }

    /**
     * 处理请求操作
     * @param request
     * @param parser
     * @param <T>
     * @return
     * @throws ServiceException
     * @throws ClientException
     */
    protected <T> T doOperation(RequestMessage request, ResponseParser<T> parser) throws ServiceException, ClientException {
        request.addHeader("Content-Type", DEFAULT_CONTENT_TYPE);
        this.requestSigner.sign(request);
        ResponseMessage response = send(request);

        try {
            return parser.parse(response.getContent());
        } catch (ResponseParseException rpe) {
            ServiceException e = new ServiceException("Failed to parse the response result.", -1, rpe.getMessage());
            logException("Unable to parse response error: ", rpe);
            throw e;
        }
    }

    /**
     * 从接入地址获取上下文路径
     * @param endpoint
     * @return
     */
    private static String getContextPath(String endpoint) {
        String[] a = endpoint.split("//");
        String domain = a[a.length - 1];
        int index = domain.lastIndexOf("/");
        if (index == -1) {
            return "";
        }
        return domain.substring(index);
    }
}
