(ns burningswell.api.countries
  "The countries of the world."
  (:require [burningswell.api.core :refer :all]
            [burningswell.api.hal :as hal]
            [burningswell.api.schemas :refer :all]
            [burningswell.api.spots :refer [enhance-spots]]
            [burningswell.db.airports :as airports]
            [burningswell.db.countries :as countries]
            [burningswell.db.ports :as ports]
            [burningswell.db.regions :as regions]
            [burningswell.db.spots :as spots]
            [burningswell.db.users :as users]
            [burningswell.http.response :refer [created ok]]
            [schema.core :as s]
            [burningswell.api.coerce :as coerce])
  (:import java.io.ByteArrayInputStream))

(set! *warn-on-reflection* true)

(defn- country-not-found
  "Return a 404 response for a country that could not be found by `id`."
  [id]
  (not-found (format "Country %s not found" id)))

(defn countries
  "List all countries."
  [{:keys [api-client db query-params]}]
  (let [params (coerce/update-location query-params)
        countries (countries/all db params)
        countries (countries/assoc-photo db countries)]
    (ok (hal/links api-client :country countries))))

(defn create-country
  "Create a new country."
  [{:keys [api-client data broker db path-params]}]
  (s/validate CreateCountry data)
  (let [country (countries/insert db data)
        country (hal/link api-client :country country)]
    (publish broker "countries.created" country)
    (created country)))

(defn country
  "Show a country."
  [{:keys [api-client db path-params]}]
  (if-let [country (countries/by-id db (:id path-params))]
    (ok (hal/link api-client :country country))
    (country-not-found (:id path-params))))

(defn delete-country
  "Delete a country."
  [{:keys [broker db path-params]}]
  (if-let [country (countries/by-id db (:id path-params))]
    (do (countries/delete db country)
        (publish broker "countries.deleted" country)
        (no-content))
    (country-not-found (:id path-params))))

(defn update-country
  "Update a country."
  [{:keys [api-client data broker db path-params]}]
  (s/validate CreateCountry data)
  (if-let [country (countries/by-id db (:id path-params))]
    (let [country (countries/update db (merge country data))
          country (countries/by-id db (:id path-params))
          country (hal/link api-client :country country)]
      (publish broker "countries.updated" country)
      (ok country))
    (country-not-found (:id path-params))))

(defn airports
  "List all airports on a country."
  [{:keys [api-client db path-params query-params]}]
  (if-let [country (countries/by-id db (:id path-params))]
    (let [airports (airports/in-country db country query-params)]
      (ok (hal/links api-client :airport airports)))
    (country-not-found (:id path-params))))

(defn image
  "Return the country image."
  [{:keys [db path-params query-params]}]
  (if-let [country (countries/by-id db (:id path-params))]
    (ok (ByteArrayInputStream.
         (countries/as-png db country query-params))
        {"cache-control" "public, max-age=86400"
         "content-type" "image/png"})
    (country-not-found (:id path-params))))

(defn ports
  "List all ports on a country."
  [{:keys [api-client db path-params query-params]}]
  (if-let [country (countries/by-id db (:id path-params))]
    (let [ports (ports/in-country db country query-params)]
      (ok (hal/links api-client :port ports)))
    (country-not-found (:id path-params))))

(defn regions
  "List all regions on a country."
  [{:keys [api-client db path-params query-params]}]
  (if-let [country (countries/by-id db (:id path-params))]
    (let [regions (regions/in-country db country query-params)
          regions (regions/assoc-photo db regions)]
      (ok (hal/links api-client :region regions)))
    (country-not-found (:id path-params))))

(defn spots
  "List all spots on a country."
  [{:keys [api-client db path-params query-params]}]
  (if-let [country (countries/by-id db (:id path-params))]
    (let [spots (spots/in-country db country query-params)
          spots (enhance-spots api-client db spots query-params)]
      (ok spots))
    (country-not-found (:id path-params))))

(defn users
  "List all users on a country."
  [{:keys [api-client db path-params query-params]}]
  (if-let [country (countries/by-id db (:id path-params))]
    (->> (users/in-country db country query-params)
         (map #(users/safe-user identity %))
         (hal/links api-client :user)
         (ok))
    (country-not-found (:id path-params))))

(set! *warn-on-reflection* false)
