package aljx.java.util.text;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import aljx.java.util.debug.Tracer;
import aljx.java.util.stream.StreamReader;
import aljx.java.util.stream.StringInputStream;

public class TextUtils {

    public static String ellipsize(String s, int maxLength) {
        return s != null && s.length() > maxLength ? s.substring(0, maxLength) + "..." : s;
    }

    public static String join(CharSequence delimiter, Iterable tokens) {
        StringBuilder sb = new StringBuilder();
        boolean firstTime = true;
        for (Object token : tokens) {
            if (firstTime) {
                firstTime = false;
            } else {
                sb.append(delimiter);
            }
            sb.append(token);
        }
        return sb.toString();
    }

    private String removeFromEnd(String source, String endStr) {
        StringBuilder sb = new StringBuilder(source);
        int lastIndexOf;
        while ((lastIndexOf = sb.lastIndexOf(endStr)) > -1 && lastIndexOf + endStr.length() == sb.length()) {
            sb.setLength(lastIndexOf);
        }
        return sb.toString();
    }

    public static String makeNotNull(String s) {
        return s != null ? s : "";
    }

    public static boolean isEmpty(String s) {
        return s == null || s.length() == 0;
    }

    public static boolean isNotEmpty(String s) {
        return !isEmpty(s);
    }

    public static boolean isAllEmpty(String... strArr) {
        for (String s : strArr) {
            if (!isEmpty(s)) {
                return false;
            }
        }
        return true;
    }

    public static boolean isAllNotEmpty(String... strArr) {
        for (String s : strArr) {
            if (isEmpty(s)) {
                return false;
            }
        }
        return true;
    }

    public static int getAfterSpaceIndex(String line, int startIndex) {
        int afterSpaceIndex;
        int spaceIndex = line.indexOf(' ', startIndex);
        if (spaceIndex >= 0) {
            try {
                afterSpaceIndex = spaceIndex + 1;
                while (line.charAt(afterSpaceIndex) == ' ') {
                    afterSpaceIndex++;
                }
                return afterSpaceIndex;
            } catch (Exception e) {
                //after space not found
                return -1;
            }
        } else {
            //space not found
            return -1;
        }
    }

    public static ArrayList<String> wordWrap(String line, int lineLength) {
        if (TextUtils.isEmpty(line)) {
            throw new IllegalArgumentException("Empty line unsupported");
        }

        if (line.contains("\n")) {
            throw new IllegalArgumentException("Line with next line chars unsupported");
        }

        ArrayList<String> lines = new ArrayList<String>();

        int startIndex = 0;
        while (true) {
            int afterSpaceIndex = TextUtils.getAfterSpaceIndex(line, startIndex);
            if (afterSpaceIndex == -1) {
                //add rest of the line
                lines.add(line.substring(startIndex));
                break;
            } else if (afterSpaceIndex - startIndex >= lineLength) {
                //first after space occurrence >= lineLength. add part of the line
                lines.add(line.substring(startIndex, afterSpaceIndex));
                startIndex = afterSpaceIndex;
            } else {
                //find most appropriate after space occurrence
                int afterSpaceIndexNext;
                while ((afterSpaceIndexNext = TextUtils.getAfterSpaceIndex(line, afterSpaceIndex)) - startIndex < lineLength && afterSpaceIndexNext != -1) {
                    afterSpaceIndex = afterSpaceIndexNext;
                }

                lines.add(line.substring(startIndex, afterSpaceIndex));
                startIndex = afterSpaceIndex;
            }
        }
        return lines;
    }

    public static List<String> splitLines(String string) {
        List<String> origTextLines = null;
        try {
            origTextLines = StreamReader.streamToStringList(new StringInputStream(string));
        } catch (IOException e) {
            Tracer.e(e);
        }
        return origTextLines;
    }

    public static String concat(List<String> text, boolean newLine) {
        StringBuilder sb = new StringBuilder();
        for (String string : text) {
            sb.append(string);
            if (newLine) {
                sb.append("\n");
            }
        }
        return sb.toString();
    }
}
