package aljx.java.util.reflection;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.concurrent.atomic.AtomicBoolean;

import aljx.java.util.debug.Tracer;

public class ReflectionUtils {

    private static final AtomicBoolean sPrintLog = new AtomicBoolean(false);

    public static void setPrintLog(boolean print) {
        sPrintLog.set(print);
    }

    public static <T> T getInstance(String className) {
        T result = null;
        try {
            Class<?> c = Class.forName(className);
            result = (T) c.newInstance();
        } catch (ClassNotFoundException e) {
            Tracer.w("Filed instantiate " + className, e);
        } catch (InstantiationException e) {
            Tracer.w("Filed instantiate " + className, e);
        } catch (IllegalAccessException e) {
            Tracer.w("Filed instantiate " + className, e);
        }

        return result;
    }

    public static <T> T getInstanceUnsafe(String className) throws Exception {
        Class<?> c = Class.forName(className);
        T result = (T) c.newInstance();
        return result;
    }

    public static Object callMethod(Object object, String methodName, Object... args) {
        return callMethod(object.getClass(), object, methodName, args);
    }

    /**
     * For static calls
     */
    public static Object callMethod(Class<?> c, String methodName, Object... args) {
        return callMethod(c, null, methodName, args);
    }

    public static Object callMethod(Class<?> c, Object obj, String methodName, Object[] args) {
        Object result = null;

        Class<?>[] argTypes = new Class[args.length];

        for (int i = 0; i < args.length; i++) {
            argTypes[i] = args[i].getClass();
        }

        result = callMethodTyped(c, obj, methodName, argTypes, args);

        return result;
    }

    public static Object callMethodVarArgs(Class<?> c, Object obj, String methodName, Object[] args, Object varArgs) {
        Object result = null;

        Class<?>[] argTypes = new Class[args.length + 1];
        Object[] allArgs = new Object[args.length + 1];

        for (int i = 0; i < args.length; i++) {
            argTypes[i] = args[i].getClass();
            allArgs[i] = args[i];
        }

        argTypes[args.length] = varArgs.getClass();
        allArgs[args.length] = varArgs;

        result = callMethodTyped(c, obj, methodName, argTypes, allArgs);

        return result;
    }

    public static Object callMethodTyped(Class<?> c, Object obj, String methodName, Class<?>[] argTypes, Object[] args) {
        if (obj != null && obj.getClass() != c) {
            throw new IllegalStateException();
        }

        Object result = null;
        try {
            if (sPrintLog.get()) {
                Tracer.print(getMethodStr(methodName, argTypes));
            }
            Method method = c.getDeclaredMethod(methodName, argTypes);
            method.setAccessible(true);
            result = method.invoke(obj, args);
        } catch (InvocationTargetException e) {
            Tracer.e(e);
        } catch (NoSuchMethodException e) {
            Tracer.e(e);
        } catch (IllegalAccessException e) {
            Tracer.e(e);
        }

        return result;
    }

    public static String getMethodStr(String methodName, Class<?>[] argTypes) {
        return String.format("%s(%s)", methodName, getTypesStr(argTypes));
    }

    public static String getTypesStr(Class<?>[] types) {
        if (types.length == 0) {
            return "";
        }
        StringBuilder result = new StringBuilder();
        appendTypeName(result, types[0]);
        for (int i = 1; i < types.length; i++) {
            result.append(',');
            appendTypeName(result, types[i]);
        }
        return result.toString();
    }

    public static void appendTypeName(StringBuilder out, Class<?> c) {
        int dimensions = 0;
        while (c.isArray()) {
            c = c.getComponentType();
            dimensions++;
        }
        out.append(c.getName());
        for (int d = 0; d < dimensions; d++) {
            out.append("[]");
        }
    }

    public static Object getField(Object object, String fieldName) {
        try {
            Class<?> c = object.getClass();
            Field field = c.getDeclaredField(fieldName);
            field.setAccessible(true);
            return field.get(object);
        } catch (IllegalAccessException e) {
            Tracer.e(e);
        } catch (NoSuchFieldException e) {
            Tracer.e(e);
        }
        return null;
    }

    public static Object getField(Class<?> c, String fieldName) {
        try {
            Field field = c.getDeclaredField(fieldName);
            field.setAccessible(true);
            return field.get(null);
        } catch (IllegalAccessException e) {
            Tracer.e(e);
        } catch (NoSuchFieldException e) {
            Tracer.e(e);
        }
        return null;
    }
}
