package aljx.java.util.file.filter;

import java.util.ArrayList;
import java.util.List;

import aljx.java.util.collection.CollectionUtils;
import aljx.java.util.config.DebugLogConfig;
import aljx.java.util.debug.Dumper;
import aljx.java.util.debug.Tracer;
import aljx.java.util.filter.Filter;

/**
 * Base class for file filters
 * <p/>
 * Use FilenameFilterWrapper to convert FileFilters to FilenameFilters
 * <p/>
 * Default <code>{@link aljx.java.util.file.filter.FileSourceFilterBase#accept(FileSource) accept(FileSource)}</code> method works
 * based on file name operation result. To use default accept logic override
 * <code>{@link aljx.java.util.file.filter.FileSourceFilterBase#fileFileNameOperation(String, String) fileFileNameOperation(String, String)}</code> and
 * <code>{@link aljx.java.util.file.filter.FileSourceFilterBase#dirFileNameOperation(String, String) dirFileNameOperation(String, String)}</code>
 * in subclasses. Or override <code>{@link aljx.java.util.file.filter.FileSourceFilterBase#accept(FileSource) accept(FileSource)}</code>
 * for implement custom accept logic based on <code>{@link aljx.java.util.file.filter.FileSourceFilterBase#fileIncludeList}</code>,
 * {@link aljx.java.util.file.filter.FileSourceFilterBase#fileExcludeList}, {@link aljx.java.util.file.filter.FileSourceFilterBase#dirIncludeList},
 * {@link aljx.java.util.file.filter.FileSourceFilterBase#dirExcludeList}.
 * <p/>
 * To include any file or directory use {@link aljx.java.util.file.filter.FileSourceFilterBase#FileSourceFilterBase()} constructor.
 * <p/>
 * To include ony files use
 * {@link aljx.java.util.file.filter.FileSourceFilterBase#FileSourceFilterBase(String...)},
 * {@link aljx.java.util.file.filter.FileSourceFilterBase#FileSourceFilterBase(java.util.List<String>)},
 * {@link aljx.java.util.file.filter.FileSourceFilterBase#FileSourceFilterBase(java.util.List<String>, java.util.List<String>)} constructors.
 * <p/>
 * To include any directory use constructor
 * <code>{@link aljx.java.util.file.filter.FileSourceFilterBase#FileSourceFilterBase(java.util.List<String>, java.util.List<String>, java.util.List<String>, java.util.List<String>)}</code>
 * with {@link aljx.java.util.file.filter.FileSourceFilterBase#dirIncludeList} == <code>null</code>
 * <p/>
 * To include any file use any constructor with {@link aljx.java.util.file.filter.FileSourceFilterBase#fileIncludeList} == <code>null</code>
 * <p/>
 * To exclude file use {@link aljx.java.util.file.filter.FileSourceFilterBase#fileExcludeList} in constructors.
 * <p/>
 * If {@link aljx.java.util.file.filter.FileSourceFilterBase#fileExcludeList} empty or <code>null</code>, no files will be excluded.
 * <p/>
 * To exclude directory  use same rules fo {@link aljx.java.util.file.filter.FileSourceFilterBase#dirExcludeList}
 */
public abstract class FileSourceFilterBase<FileSource> implements Filter<FileSource> {
    private static final boolean PRINT_LOG = DebugLogConfig.PRINT_FILE_FILTER_LOGIC;
    List<String> fileIncludeList = null;
    List<String> fileExcludeList = null;

    List<String> dirIncludeList = null;
    List<String> dirExcludeList = null;

    public FileSourceFilterBase() {

    }

    public FileSourceFilterBase(String... fileIncludeArr) {
        this(false, CollectionUtils.toList(fileIncludeArr), null);
    }

    public FileSourceFilterBase(List<String> fileIncludeList) {
        this(false, fileIncludeList, null);
    }

    public FileSourceFilterBase(List<String> fileIncludeList, List<String> fileExcludeList) {
        this(false, fileIncludeList, fileExcludeList);
    }

    public FileSourceFilterBase(boolean acceptDirs, String... fileIncludeArr) {
        this(acceptDirs, CollectionUtils.toList(fileIncludeArr), null);
    }

    public FileSourceFilterBase(boolean acceptDirs, List<String> fileIncludeList) {
        this(acceptDirs, fileIncludeList, null);
    }

    public FileSourceFilterBase(boolean acceptDirs, List<String> fileIncludeList, List<String> fileExcludeList) {
        this(fileIncludeList, fileExcludeList, acceptDirs ? null : new ArrayList<String>(), null);
    }

    public FileSourceFilterBase(List<String> fileIncludeList, List<String> fileExcludeList,
                                List<String> dirIncludeList, List<String> dirExcludeList) {
        if (fileIncludeList != null) {
            this.fileIncludeList = new ArrayList<String>();
            this.fileIncludeList.addAll(fileIncludeList);
        }

        if (fileExcludeList != null) {
            this.fileExcludeList = new ArrayList<String>();
            this.fileExcludeList.addAll(fileExcludeList);
        }

        if (dirIncludeList != null) {
            this.dirIncludeList = new ArrayList<String>();
            this.dirIncludeList.addAll(dirIncludeList);
        }

        if (dirExcludeList != null) {
            this.dirExcludeList = new ArrayList<String>();
            this.dirExcludeList.addAll(dirExcludeList);
        }
    }

    @Override
    public boolean accept(FileSource fileSource) {
        if (PRINT_LOG) {
            Tracer.print("===========");
            Tracer.print("fileSource = " + fileSource);
            Tracer.print("filter rules:\nfile include list = " + Dumper.dump(fileIncludeList)
                    + "\nfile exclude list = " + Dumper.dump(fileExcludeList)
                    + "\ndir include list = " + Dumper.dump(dirIncludeList)
                    + "\ndir exclude list = " + Dumper.dump(dirExcludeList));

        }

        if (fileSource == null) {
            if (PRINT_LOG) {
                Tracer.print("fileSource source == null return false");
            }

            return false;
        }

        String name = getName(fileSource);
        boolean isFile = isFile(fileSource);
        boolean isDirectory = isDirectory(fileSource);

        if (isFile) {

            if (PRINT_LOG) {
                Tracer.print("isFile");
            }

            if (fileExcludeList != null) {
                for (String val : fileExcludeList) {
                    if (PRINT_LOG) {
                        Tracer.print("check exclude fileSource. cur = " + val + " name = " + name);
                    }
                    if (fileFileNameOperation(val, name)) {
                        if (PRINT_LOG) {
                            Tracer.print("in exclude list. name = " + name + " return false");
                        }
                        return false;
                    }
                }
            }

            if (fileIncludeList == null) {
                if (PRINT_LOG) {
                    Tracer.print("include. return true");
                }
                return true;
            }

            for (String val : fileIncludeList) {
                if (PRINT_LOG) {
                    Tracer.print("check include fileSource. cur = " + val + " name = " + name);
                }
                if (fileFileNameOperation(val, name)) {
                    if (PRINT_LOG) {
                        Tracer.print("in include list. name = " + name + " return true");
                    }
                    return true;
                }
            }

            if (PRINT_LOG) {
                Tracer.print("not in include file list. return false");
            }

            return false;
        }

        if (isDirectory) {

            if (PRINT_LOG) {
                Tracer.print("isDirectory");
            }

            if (dirExcludeList != null) {
                for (String val : dirExcludeList) {

                    if (PRINT_LOG) {
                        Tracer.print("check exclude dir. cur = " + val + " name = " + name);
                    }

                    if (dirFileNameOperation(val, name)) {
                        if (PRINT_LOG) {
                            Tracer.print("in exclude list. name = " + name + " return false");
                        }
                        return false;
                    }
                }
            }

            if (dirIncludeList == null) {
                if (PRINT_LOG) {
                    Tracer.print("include. return true");
                }
                return true;
            }

            for (String val : dirIncludeList) {

                if (PRINT_LOG) {
                    Tracer.print("check include dir. cur = " + val + " name = " + name);
                }

                if (dirFileNameOperation(val, name)) {

                    if (PRINT_LOG) {
                        Tracer.print("in include list. name = " + name + " return true");
                    }

                    return true;
                }
            }

            if (PRINT_LOG) {
                Tracer.print("not in include dir list. return false");
            }

            return false;
        }

        if (PRINT_LOG) {
            Tracer.print("not file not dir. return false");
        }
        return false;
    }

    protected abstract boolean isDirectory(FileSource fileSource);

    protected abstract boolean isFile(FileSource fileSource);

    protected abstract String getName(FileSource fileSource);

    protected abstract boolean fileFileNameOperation(String ruleVal, String filename);

    protected abstract boolean dirFileNameOperation(String ruleVal, String filename);
}