package aljx.java.util.file;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.charset.Charset;
import java.util.concurrent.atomic.AtomicReference;

import aljx.java.util.debug.Tracer;

public class LogSaveHelper {
    public static final int ONE_K = 1024;
    public static final int TEN_K = 10 * ONE_K;
    public static final long ONE_MB = 1024 * ONE_K;
    private static final long EIGHT_MB = 8 * ONE_MB;
    private static final AtomicReference<Long> sMaxLogFileSize = new AtomicReference<Long>(EIGHT_MB);
    private static FileOutputStream mLogFileStream;
    private static File logFile;
    private static final AtomicReference<String> sLogFileName = new AtomicReference<String>();
    private static final AtomicReference<String> sOldLogFileName = new AtomicReference<String>();
    private static final AtomicReference<File> sLogFileDir = new AtomicReference<File>();

    public static void setSettings(File logFileDir, String logFileName, String oldLogFileName, long maxFileSize) {
        sMaxLogFileSize.set(maxFileSize);
        sLogFileName.set(logFileName);
        sOldLogFileName.set(oldLogFileName);
        sLogFileDir.set(logFileDir);
    }

    public static boolean write(byte[] buffer, int byteOffset, int byteCount) throws IOException {
        FileOutputStream logFileStream;
        if ((logFileStream = LogSaveHelper.getLogFileStream(false)) != null) {
            logFileStream.write(buffer, byteOffset, byteCount);
            logFileStream.flush();
        }
        return true;
    }

    public static void write(String s) throws IOException {
        byte[] bytes = s.getBytes(Charset.forName("UTF-8"));
        write(bytes, 0, bytes.length);
    }

    public static FileOutputStream getLogFileStream(boolean prepare) {
        if (prepare || (logFile != null && logFile.length() > sMaxLogFileSize.get())) {
            closeLogFileStream();
            prepareLogFile();
            initLogWriter();
        }

        if (mLogFileStream == null || logFile == null) {
            initLogWriter();
        }
        return mLogFileStream;
    }

    public static void closeLogFileStream() {
        if (mLogFileStream != null) {
            try {
                mLogFileStream.close();
                mLogFileStream = null;
                logFile = null;
            } catch (IOException e) {
                mLogFileStream = null;
                logFile = null;
                Tracer.e("error closing output stream", e);
            }
        }
    }

    private static File prepareLogFile() {
        File f = null;
        try {
            f = new File(sLogFileDir.get(), sLogFileName.get());
            if (f.exists()) {
                File from = new File(sLogFileDir.get(), sLogFileName.get());
                File to = new File(sLogFileDir.get(), sOldLogFileName.get());
                if (to.exists()) {
                    to.delete();
                }
                from.renameTo(to);
            }
            f = new File(sLogFileDir.get(), sLogFileName.get());
            f.createNewFile();
        } catch (IOException e) {
            Tracer.e(e);
        }
        return f;
    }

    private static File getLogFile() {
        File f = null;
        try {
            f = new File(sLogFileDir.get(), sLogFileName.get());
            if (!f.exists()) {
                f.createNewFile();
            }
        } catch (IOException e) {
            Tracer.e(e);
        }
        return f;
    }

    private static void initLogWriter() {
        try {
            logFile = LogSaveHelper.getLogFile();
            mLogFileStream = new FileOutputStream(logFile, true);
        } catch (IOException e) {
            Tracer.e(e);
        }
    }
}
