package aljx.java.util.file;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileFilter;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;

import aljx.java.util.file.filter.NotFileNotDirectory;

/**
 * @author Alex Andriichenko
 */
public class FileAnalyzer {

    private boolean printLog;

    public FileAnalyzer() {
        printLog = false;
    }

    public FileAnalyzer(boolean printLog) {
        this.printLog = printLog;
    }

    private final ArrayList<FileInfo> mFileList = new ArrayList<FileInfo>();

    public void clear() {
        mFileList.clear();
    }

    public int size() {
        return mFileList.size();
    }

    public void findFiles(File file) {
        findFiles(file, null, null);
    }

    public void findNotFileNotDirectory(File file) {
        findFiles(file, new NotFileNotDirectory(), null);
    }

    public void findFiles(File file, FileFilter addFilter, FileFilter listFilter) {
        ArrayList<File> list = FileInfoSearcher.getAllInnerFiles(file, addFilter, listFilter);

        if (printLog) {
            System.out.println("\n" + list.size() + " files added");
            System.out.println("Get file info...");
        }

        for (File f : list) {
            mFileList.add(new FileInfo(f));
        }
    }

    public enum FileSortType {

        SIZE, NAME, FILE_TYPE, ALPHABETIC, FILE_TYPE_AND_DATE, NONE;
    }

    public void sort(FileSortType sortType) {
        Comparator<FileInfo> comparator = null;
        switch (sortType) {
            case SIZE:
                comparator = new SizeFileInfoComparator();
                break;
            case NAME:
                comparator = new NameFileInfoComparator();
                break;
            case FILE_TYPE:
                comparator = new TypeFileInfoComparator();
                break;
            case ALPHABETIC:
                comparator = new AlphabeticComparator();
                break;
            case FILE_TYPE_AND_DATE:
                comparator = new TypeAndDateFileInfoComparator();
                break;
            case NONE:
            default:
                return;
        }

        if (printLog) {
            System.out.println("comparator = " + comparator.getClass());
        }

        Collections.sort(mFileList, comparator);
    }

    public final class AlphabeticComparator implements Comparator<FileInfo> {

        @Override
        public int compare(FileInfo fi1, FileInfo fi2) {
            int compareResult = Boolean.compare(fi1.isDirectory, fi2.isDirectory);

            if (compareResult == 0) {
                compareResult = fi1.name.compareTo(fi2.name);
            }
            return compareResult;
        }
    }

    public final class SizeFileInfoComparator implements Comparator<FileInfo> {

        @Override
        public int compare(FileInfo fi1, FileInfo fi2) {
            int compareResult = Boolean.compare(fi1.isDirectory, fi2.isDirectory);

            if (compareResult == 0) {
                compareResult = Long.compare(fi2.fileSize, fi1.fileSize);
            }

            if (compareResult == 0) {
                compareResult = fi1.name.compareTo(fi2.name);
            }

            if (compareResult == 0) {
                compareResult = fi1.path.compareTo(fi2.path);
            }

            return compareResult;
        }
    }

    public final class NameFileInfoComparator implements Comparator<FileInfo> {

        @Override
        public int compare(FileInfo fi1, FileInfo fi2) {
            int compareResult = Boolean.compare(fi1.isDirectory, fi2.isDirectory);

            if (compareResult == 0) {
                compareResult = fi1.name.compareTo(fi2.name);
            }

            if (compareResult == 0) {
                compareResult = fi1.path.compareTo(fi2.path);
            }

            if (compareResult == 0) {
                compareResult = Long.compare(fi2.fileSize, fi1.fileSize);
            }

            return compareResult;
        }
    }

    public final class TypeFileInfoComparator implements Comparator<FileInfo> {

        @Override
        public int compare(FileInfo fi1, FileInfo fi2) {
            int compareResult = Boolean.compare(fi1.isDirectory, fi2.isDirectory);

            if (compareResult == 0 && !fi1.isDirectory && !fi2.isDirectory) {
                String[] fi1SplittedArr = fi1.name.split("\\.");
                String[] fi2SplittedArr = fi2.name.split("\\.");

                if (fi1SplittedArr.length == 0) {
                    throw new RuntimeException("name = " + fi1.name);
                }

                if (fi2SplittedArr.length == 0) {
                    throw new RuntimeException("name = " + fi2.name);
                }

                if (fi1SplittedArr.length > 1 && fi2SplittedArr.length > 1) {
                    compareResult = fi1SplittedArr[fi1SplittedArr.length - 1].compareTo(fi2SplittedArr[fi2SplittedArr.length - 1]);
                } else if (fi1SplittedArr.length >= 1 && fi2SplittedArr.length >= 1) {
                    compareResult = Integer.compare(fi1SplittedArr.length, fi2SplittedArr.length);
                } else {
                    throw new IllegalStateException("Somthing wrong in compare logic");
                }
            }

            if (compareResult == 0) {
                compareResult = fi1.name.compareTo(fi2.name);
            }

            if (compareResult == 0) {
                compareResult = fi1.path.compareTo(fi2.path);
            }

            if (compareResult == 0) {
                compareResult = Long.compare(fi2.fileSize, fi1.fileSize);
            }

            return compareResult;
        }
    }

    public final class TypeAndDateFileInfoComparator implements Comparator<FileInfo> {

        @Override
        public int compare(FileInfo fi1, FileInfo fi2) {
            int compareResult = Boolean.compare(fi1.isDirectory, fi2.isDirectory);

            if (compareResult == 0 && !fi1.isDirectory && !fi2.isDirectory) {
                String[] fi1SplittedArr = fi1.name.split("\\.");
                String[] fi2SplittedArr = fi2.name.split("\\.");

                if (fi1SplittedArr.length == 0) {
                    throw new RuntimeException("name = " + fi1.name);
                }

                if (fi2SplittedArr.length == 0) {
                    throw new RuntimeException("name = " + fi2.name);
                }

                if (fi1SplittedArr.length > 1 && fi2SplittedArr.length > 1) {
                    compareResult = fi1SplittedArr[fi1SplittedArr.length - 1].compareTo(fi2SplittedArr[fi2SplittedArr.length - 1]);
                } else if (fi1SplittedArr.length >= 1 && fi2SplittedArr.length >= 1) {
                    compareResult = Integer.compare(fi1SplittedArr.length, fi2SplittedArr.length);
                } else {
                    throw new IllegalStateException("Somthing wrong in compare logic");
                }
            }

            if (compareResult == 0) {
                compareResult = Long.compare(fi1.lastModified, fi2.lastModified);
            }

            if (compareResult == 0) {
                compareResult = fi1.name.compareTo(fi2.name);
            }

            if (compareResult == 0) {
                compareResult = fi1.path.compareTo(fi2.path);
            }

            if (compareResult == 0) {
                compareResult = Long.compare(fi2.fileSize, fi1.fileSize);
            }

            return compareResult;
        }
    }

    public void saveToFile(File file, String header) throws IOException {
        if (!file.exists()) {
            file.createNewFile();
        }

        BufferedWriter bufferedWriter = new BufferedWriter(new FileWriter(file));
        bufferedWriter.write(header);
        long l = 0;
        FileInfo prevFi = null;
        for (FileInfo fi : mFileList) {
            bufferedWriter.write(fi.getFileInfoString(prevFi));
            prevFi = fi;
            l++;

            if (printLog) {
                if (l % 1000 == 0) {
                    System.out.print('.');
                }

                if (l % 100000 == 0) {
                    System.out.println();
                }
            }
        }

        bufferedWriter.close();

        if (printLog) {
            System.out.println("\n " + l + " strings saved");
        }
    }

    public void saveToFileBriefly(File file, String header) throws IOException {
        if (!file.exists()) {
            file.createNewFile();
        }

        BufferedWriter bufferedWriter = new BufferedWriter(new FileWriter(file));
        bufferedWriter.write(header);
        long l = 0;
        for (FileInfo fi : mFileList) {
            bufferedWriter.write(fi.name + "\n");
            l++;

            if (printLog) {
                if (l % 1000 == 0) {
                    System.out.print('.');
                }

                if (l % 100000 == 0) {
                    System.out.println();
                }
            }
        }

        bufferedWriter.close();

        if (printLog) {
            System.out.println("\n " + l + " strings saved");
        }
    }
}
