package aljx.java.util.debug;

import java.net.URI;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.zip.ZipEntry;

public class DefaultDumpExtension extends Dumper.DumpExtension {

    @Override
    public String dumpImpl(Object value, ArrayList dumpedObjects) {
        if (!(canDump(value))) {
            throw new IllegalArgumentException(value.getClass().getName()
                    + " unsupported for dump");
        }

        if (value instanceof byte[]) {
            return dumpImpl((byte[]) value, dumpedObjects);
        } else if (value instanceof short[]) {
            return dumpImpl((short[]) value, dumpedObjects);
        } else if (value instanceof int[]) {
            return dumpImpl((int[]) value, dumpedObjects);
        } else if (value instanceof long[]) {
            return dumpImpl((long[]) value, dumpedObjects);
        } else if (value instanceof float[]) {
            return dumpImpl((float[]) value, dumpedObjects);
        } else if (value instanceof double[]) {
            return dumpImpl((double[]) value, dumpedObjects);
        } else if (value instanceof char[]) {
            return dumpImpl((char[]) value, dumpedObjects);
        } else if (value instanceof boolean[]) {
            return dumpImpl((boolean[]) value, dumpedObjects);
        } else if (value instanceof Object[]) {
            return dumpImpl((Object[]) value, dumpedObjects);
        } else if (value instanceof Collection<?>) {
            return dumpImpl((Collection<?>) value, dumpedObjects);
        } else if (value instanceof Map) {
            return dumpImpl((Map) value, dumpedObjects);
        } else if (value instanceof Map.Entry) {
            return dumpImpl((Map.Entry) value, dumpedObjects);
        } else if (value instanceof Iterator) {
            return dumpImpl((Iterator) value, dumpedObjects);
        } else if (value instanceof Throwable) {
            return dumpImpl((Throwable) value, dumpedObjects);
        } else if (value instanceof URI) {
            return dumpImpl((URI) value, dumpedObjects);
        } else if (value instanceof ZipEntry) {
            return dumpImpl((ZipEntry) value, dumpedObjects);
        } else if (value instanceof Properties) {
            return dumpImpl((Properties) value, dumpedObjects);
        } else if (value instanceof Thread) {
            return dumpImpl((Thread) value, dumpedObjects);
        } else if (value instanceof ThreadGroup) {
            return dumpImpl((ThreadGroup) value, dumpedObjects);
        }

        throw new IllegalStateException("Failed dump " + value.getClass().getName());
    }

    @Override
    public boolean canDump(Object value) {
        if (value instanceof byte[]) {
            return true;
        } else if (value instanceof short[]) {
            return true;
        } else if (value instanceof int[]) {
            return true;
        } else if (value instanceof long[]) {
            return true;
        } else if (value instanceof float[]) {
            return true;
        } else if (value instanceof double[]) {
            return true;
        } else if (value instanceof char[]) {
            return true;
        } else if (value instanceof boolean[]) {
            return true;
        } else if (value instanceof Object[]) {
            return true;
        } else if (value instanceof Collection<?>) {
            return true;
        } else if (value instanceof Map) {
            return true;
        } else if (value instanceof Map.Entry) {
            return true;
        } else if (value instanceof Iterator) {
            return true;
        } else if (value instanceof Throwable) {
            return true;
        } else if (value instanceof URI) {
            return true;
        } else if (value instanceof ZipEntry) {
            return true;
        } else if (value instanceof Properties) {//extends Map
            return true;
        } else if (value instanceof Thread) {
            return true;
        } else if (value instanceof ThreadGroup) {
            return true;
        }
        return false;
    }

    private static String dumpImpl(byte[] array, ArrayList dumpedObjects) {
        return Arrays.toString(array);
    }

    private static String dumpImpl(short[] array, ArrayList dumpedObjects) {
        return Arrays.toString(array);
    }

    private static String dumpImpl(int[] array, ArrayList dumpedObjects) {
        return Arrays.toString(array);
    }

    private static String dumpImpl(long[] array, ArrayList dumpedObjects) {
        return Arrays.toString(array);
    }

    private static String dumpImpl(float[] array, ArrayList dumpedObjects) {
        return Arrays.toString(array);
    }

    private static String dumpImpl(double[] array, ArrayList dumpedObjects) {
        return Arrays.toString(array);
    }

    private static String dumpImpl(char[] array, ArrayList dumpedObjects) {
        return Arrays.toString(array);
    }

    private static String dumpImpl(boolean[] array, ArrayList dumpedObjects) {
        return Arrays.toString(array);
    }

    private String dumpImpl(Object[] array, ArrayList dumpedObjects) {
        return dump(Arrays.asList(array), dumpedObjects);
    }

    private String dumpImpl(Collection<?> collection, ArrayList dumpedObjects) {
        return dump(collection.iterator(), dumpedObjects);
    }

    private String dumpImpl(Map<?, ?> map, ArrayList dumpedObjects) {
        return map.getClass().getSimpleName() + " " + dump(map.entrySet().iterator(), dumpedObjects);
    }

    private String dumpImpl(Map.Entry<?, ?> entry, ArrayList dumpedObjects) {
        return dump(entry.getKey(), dumpedObjects) + "=" + dump(entry.getValue(), dumpedObjects);
    }

    private String dumpImpl(Iterator iterator, ArrayList dumpedObjects) {
        StringBuilder sb = new StringBuilder();
        sb.append('[');

        if (iterator.hasNext()) {
            sb.append(dump(iterator.next(), dumpedObjects));

            while (iterator.hasNext()) {
                sb.append(", ").append(dump(iterator.next(), dumpedObjects));
            }
        }

        sb.append(']');
        return sb.toString();
    }

    private String dumpImpl(Throwable ex, ArrayList dumpedObjects) {
        StringBuilder sb = new StringBuilder();
        String info = null;
        try {
            info = ex.toString();
        } catch (Throwable e) {

        }

        sb.append(info).append('\n');

        StackTraceElement[] stackTrace = null;
        try {
            stackTrace = ex.getStackTrace();
        } catch (Throwable e) {

        }

        if (stackTrace == null) {
            sb.append("null").append('\n');
        } else {
            for (StackTraceElement stackTraceElement : stackTrace) {
                sb.append(stackTraceElement.toString()).append('\n');
            }
        }

        Throwable cause = null;
        try {
            cause = ex.getCause();
        } catch (Throwable e) {

        }

        if (cause != null && cause != ex) {
            sb.append("Caused by: ").append('\n').append(dump(cause, dumpedObjects));
        }
        return sb.toString();
    }

    private static String dumpImpl(URI uri, ArrayList dumpedObjects) {
        String authority = uri.getAuthority();
        String fragment = uri.getFragment();
        String host = uri.getHost();
        String path = uri.getPath();
        int port = uri.getPort();
        String query = uri.getQuery();

        String encodedAuthority = uri.getRawAuthority();
        String encodedFragment = uri.getRawFragment();
        String encodedPath = uri.getRawPath();
        String encodedQuery = uri.getRawQuery();
        String encodedSchemeSpecificPart = uri.getRawSchemeSpecificPart();
        String encodedUserInfo = uri.getRawUserInfo();

        String scheme = uri.getScheme();
        String schemeSpecificPart = uri.getSchemeSpecificPart();
        String userInfo = uri.getUserInfo();

        String toASCIIString = uri.toASCIIString();
        String normalize = uri.normalize().toString();

        String normalizeScheme = null;
        String lastPathSegment = null;
        List<String> pathSegments = null;
        boolean isAbsolute = uri.isAbsolute();
        boolean isOpaque = uri.isOpaque();
        Boolean isHierarchical = null;
        Boolean isRelative = null;
        Set<String> queryParameterNames = null;

        final StringBuilder sb = new StringBuilder();
        sb.append(uri);
        sb.append("\n");
        sb.append(" authority = ").append(authority);
        sb.append("\n");
        sb.append(" encodedAuthority = ").append(encodedAuthority);
        sb.append("\n");
        sb.append(" encodedFragment = ").append(encodedFragment);
        sb.append("\n");
        sb.append(" encodedPath = ").append(encodedPath);
        sb.append("\n");
        sb.append(" encodedQuery = ").append(encodedQuery);
        sb.append("\n");
        sb.append(" encodedSchemeSpecificPart = ").append(
                encodedSchemeSpecificPart);
        sb.append("\n");
        sb.append(" encodedUserInfo = ").append(encodedUserInfo);
        sb.append("\n");
        sb.append(" fragment = ").append(fragment);
        sb.append("\n");
        sb.append(" host = ").append(host);
        sb.append("\n");
        sb.append(" lastPathSegment = ").append(lastPathSegment);
        sb.append("\n");
        sb.append(" path = ").append(path);
        sb.append("\n");
        sb.append(" pathSegments = ").append(pathSegments);
        sb.append("\n");
        sb.append(" port = ").append(port);
        sb.append("\n");
        sb.append(" query = ").append(query);
        sb.append("\n");
        sb.append(" queryParameterNames = ").append(queryParameterNames);
        sb.append("\n");
        sb.append(" scheme = ").append(scheme);
        sb.append("\n");
        sb.append(" schemeSpecificPart = ").append(schemeSpecificPart);
        sb.append("\n");
        sb.append(" userInfo = ").append(userInfo);
        sb.append("\n");

        sb.append(" toASCIIString = ").append(toASCIIString);
        sb.append("\n");
        sb.append(" normalize = ").append(normalize);
        sb.append("\n");
        sb.append(" normalizeScheme = ").append(normalizeScheme);
        sb.append("\n");
        sb.append(" isAbsolute = ").append(isAbsolute);
        sb.append("\n");
        sb.append(" isOpaque = ").append(isOpaque);
        sb.append("\n");
        sb.append(" isHierarchical = ").append(isHierarchical);
        sb.append("\n");
        sb.append(" isRelative = ").append(isRelative);
        sb.append("\n");
        return sb.toString();
    }

    private String dumpImpl(ZipEntry zipEntry, ArrayList dumpedObjects) {
        StringBuilder sb = new StringBuilder();
        sb.append("ZipEntry [");
        sb.append("\n").append("class = " + zipEntry.getClass());
        sb.append("\n").append("name = " + zipEntry.getName());
        sb.append("\n").append("isDirectory = " + zipEntry.isDirectory());
        sb.append("\n").append("comment = " + zipEntry.getComment());
        sb.append("\n").append("compressedSize = " + zipEntry.getCompressedSize());
        sb.append("\n").append("crc = " + zipEntry.getCrc());
//        sb.append("\n").append("creationTime = " + zipEntry.getCreationTime());
        sb.append("\n").append("extra = " + zipEntry.getExtra());
//        sb.append("\n").append("lastAccessTime = " + zipEntry.getLastAccessTime());
//        sb.append("\n").append("lastModifiedTime = " + zipEntry.getLastModifiedTime());
        sb.append("\n").append("method = " + zipEntry.getMethod());
        sb.append("\n").append("size = " + zipEntry.getSize());
        sb.append("\n").append("time = " + zipEntry.getTime());
        sb.append("\n").append("]");
        return sb.toString();
    }

    private String dumpImpl(Properties properties, ArrayList dumpedObjects) {
        Set<String> keys = properties.stringPropertyNames();
        HashMap<String, String> hm = new HashMap<String, String>();
        for (String key : keys) {
            hm.put(key, properties.getProperty(key));
        }
        return String.format("Properties [%s]", dump(hm, dumpedObjects));
    }

    private String dumpImpl(Thread object, ArrayList dumpedObjects) {
        if (object == null) {
            return null;
        }
        return new StringBuilder()
                .append(object.getClass().getName())
                .append("[").append("@").append(Integer.toHexString(object.hashCode()))
                .append(", id=").append(object.getId())
                .append(", name=").append(object.getName())
                .append(", getPriority = ").append(object.getPriority())
                .append(", getState = ").append(object.getState())
                .append(", isDaemon = ").append(object.isDaemon())
                .append(", isAlive = ").append(object.isAlive())
                .append(", isInterrupted = ").append(object.isInterrupted())
//                .append(", getContextClassLoader = ").append("" + object.getContextClassLoader())
                .append(", getThreadGroup = ").append(dump(object.getThreadGroup(), dumpedObjects))
                .append("]").toString() + object.getId();
    }

    private String dumpImpl(ThreadGroup object, ArrayList dumpedObjects) {
        return new StringBuilder()
                .append(object.getClass().getName())
                .append("[").append("@").append(Integer.toHexString(object.hashCode()))
                .append(", name=").append(object.getName())
                .append(", maxPriority=").append(object.getMaxPriority())
                .append(", isDaemon = ").append(object.isDaemon())
                .append("]").toString();
    }
}
